/*
 * jdcolor.c
 *
 * Copyright (C) 1991, 1992, 1993, Thomas G. Lane.
 * This file is part of the Independent JPEG Group's software.
 * For conditions of distribution and use, see the accompanying README file.
 *
 * This file contains output colorspace conversion routines.
 * These routines are invoked via the methods color_convert
 * and colorout_init/term.
 */

/*#include "jinclude.h"*/
#include "viewdef.h"
#include "extern.h"

#include <dos.h>

/* added stuff from DVPEG */
extern unsigned char red, green, blue;		/* colors for saving full 24 bit resolution */

unsigned int FAR * int_ptr;
unsigned int FAR * tmp_ptr;		/* point to an integer to read the structure faster ie words not bytes */
unsigned int FAR * int_ptr2;
unsigned int FAR * tmp_ptr2;		/* point to an integer to read the structure faster ie words not bytes */



/**************** YCbCr -> RGB conversion: most common case **************/

/*
 * YCbCr is defined per CCIR 601-1, except that Cb and Cr are
 * normalized to the range 0..MAXJSAMPLE rather than -0.5 .. 0.5.
 * The conversion equations to be implemented are therefore
 *	R = Y                + 1.40200 * Cr
 *	G = Y - 0.34414 * Cb - 0.71414 * Cr
 *	B = Y + 1.77200 * Cb
 * where Cb and Cr represent the incoming values less MAXJSAMPLE/2.
 * (These numbers are derived from TIFF 6.0 section 21, dated 3-June-92.)
 *
 * To avoid floating-point arithmetic, we represent the fractional constants
 * as integers scaled up by 2^16 (about 4 digits precision); we have to divide
 * the products by 2^16, with appropriate rounding, to get the correct answer.
 * Notice that Y, being an integral input, does not contribute any fraction
 * so it need not participate in the rounding.
 *
 * For even more speed, we avoid doing any multiplications in the inner loop
 * by precalculating the constants times Cb and Cr for all possible values.
 * For 8-bit JSAMPLEs this is very reasonable (only 256 entries per table);
 * for 12-bit samples it is still acceptable.  It's not very reasonable for
 * 16-bit samples, but if you want lossless storage you shouldn't be changing
 * colorspace anyway.
 * The Cr=>R and Cb=>B values can be rounded to integers in advance; the
 * values for the G calculation are left scaled up, since we must add them
 * together before rounding.
 */

#ifdef SIXTEEN_BIT_SAMPLES
#define SCALEBITS	14	/* avoid overflow */
#else
#define SCALEBITS	16	/* speedier right-shift on some machines */
#endif
#define ONE_HALF	((INT32) 1 << (SCALEBITS-1))
#define FIX(x)		((INT32) ((x) * (1L<<SCALEBITS) + 0.5))

static int * Cr_r_tab;		/* => table for Cr to R conversion */
static int * Cb_b_tab;		/* => table for Cb to B conversion */
static int * Cb_tab;
/*		static INT32 * Cr_g_tab;	/* => table for Cr to G conversion */
/*		static INT32 * Cb_g_tab;	/* => table for Cb to G conversion */



/*
 * do tinting if the option is turned on
 */

void near tint_graphics(void)
{
int red_new, green_new, blue_new;	/* for color tinting */

red_new = new_tint(red, tint_factor_red) + red;
green_new = new_tint(green, tint_factor_green) + green;
blue_new = new_tint(blue, tint_factor_blue) + blue;

if (((red_new | green_new | blue_new) & 0xff00) == 0){
	blue = blue_new;
	green = green_new;
	red = red_new;
	}
}


/*
 * Initialize for colorspace conversion.
 */

METHODDEF void
ycc_rgb_init (decompress_info_ptr cinfo)
{
  INT32 i, x2;
  SHIFT_TEMPS

  Cr_r_tab = (int *) (*cinfo->emethods->alloc_small)
				((MAXJSAMPLE+1) * SIZEOF(int));
  Cb_b_tab = (int *) (*cinfo->emethods->alloc_small)
				((MAXJSAMPLE+1) * SIZEOF(int));
/*  Cr_g_tab = (INT32 *) (*cinfo->emethods->alloc_small)
				((MAXJSAMPLE+1) * SIZEOF(INT32));
  Cb_g_tab = (INT32 *) (*cinfo->emethods->alloc_small)
				((MAXJSAMPLE+1) * SIZEOF(INT32));*/

  Cb_tab = (int *) (*cinfo->emethods->alloc_small)
				((385) * SIZEOF(int));

  for (i = 0; i <= MAXJSAMPLE; i++) {
	 /* i is the actual input pixel value, in the range 0..MAXJSAMPLE */
	 /* The Cb or Cr value we are thinking of is x = i - MAXJSAMPLE/2 */
	 x2 = 2*i - MAXJSAMPLE;	/* twice x */
	 /* Cr=>R value is nearest int to 1.40200 * x */
	 Cr_r_tab[i] = (int)
			 RIGHT_SHIFT(FIX(1.40200/2) * x2 + ONE_HALF, SCALEBITS);
	 /* Cb=>B value is nearest int to 1.77200 * x */
	 Cb_b_tab[i] = (int)
			 RIGHT_SHIFT(FIX(1.77200/2) * x2 + ONE_HALF, SCALEBITS);
	 /* Cr=>G value is scaled-up -0.71414 * x */
/*	 Cr_g_tab[i] = (- FIX(0.71414/2)) * x2;
	 /* Cb=>G value is scaled-up -0.34414 * x */
	 /* We also add in ONE_HALF so that need not do it in inner loop */
/*	 Cb_g_tab[i] = (- FIX(0.34414/2)) * x2 + ONE_HALF;*/
  }
for (i=0; i < 385; i++){
	Cb_tab[i] = (int)RIGHT_SHIFT((-FIX(0.71414)) * (i-192) + ONE_HALF, SCALEBITS);
	}
}

#ifndef small_viewer

/*
 * Convert some rows of samples to the output colorspace. ==> 15 bit
 */

METHODDEF void
ycc_rgb_15_convert (decompress_info_ptr cinfo, int num_rows, long num_cols,
		 JSAMPIMAGE input_data, JSAMPIMAGE output_data)
{
#ifdef SIXTEEN_BIT_SAMPLES
  register INT32 y;
  register UINT16 cb, cr;
#else
  register int cg, cb, cr;
#endif
  register JSAMPROW inptr0, inptr1, inptr2;
  register int col, cols;
  /* copy these pointers into registers if possible */
  register JSAMPLE * range_limit;
  register int * Crrtab = Cr_r_tab;
  register int * Cbbtab = Cb_b_tab;
/*
  register INT32 * Crgtab = Cr_g_tab;
  register INT32 * Cbgtab = Cb_g_tab;
*/
  register int * cbtab = Cb_tab;
  int row;
  SHIFT_TEMPS

cols = num_cols;

for (row = 0; row < num_rows; row++) {
	if (!enable_pan)
		read_row = 0;
	tmp_ptr = int_ptr = (unsigned int FAR *) *((*cinfo->emethods->access_big_sarray)
			(raw_pic_ptr, read_row++, TRUE));
	inptr2 = input_data[2][row];
	inptr1 = input_data[1][row];
	inptr0 = input_data[0][row];
	for (col = 0; col < cols; col++) {
		cb = GETJSAMPLE(inptr1[col]);
		cr = GETJSAMPLE(inptr2[col]);
		cg = cbtab[cr + (cb >> 1)];
		cb = Cbbtab[cb];
		cr = Crrtab[cr];
		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr0[col]);
		red = range_limit[cb];
		green = range_limit[cg];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		* int_ptr++ = (red >> 3) + ((green << 2) & 0x3e0) + ((blue << 7) & 0xfc00);
		}
	draw_line(gr_row, (unsigned char FAR *)tmp_ptr, line_buffer_ptr);
	gr_row++;
	}
}



/*
 * Convert some rows of samples to the output colorspace. ==> 15 bit
 * assume un-upsampled data!
 *
 * more speedup is possible if I duplicate for the row below at the same time
 */

METHODDEF void
ycc_rgb_fst_15_convert (decompress_info_ptr cinfo, int num_rows, long num_cols,
		 JSAMPIMAGE input_data, JSAMPIMAGE output_data)
{
  register int cb, cr, cg;
  register JSAMPROW inptr0, inptr1, inptr2, inptr00;
  register int col, cols;
  register JSAMPLE * range_limit;
  int * Crrtab = Cr_r_tab;
  int * Cbbtab = Cb_b_tab;
  int * cbtab = Cb_tab;
  int row;

cols = num_cols >> 1;


for (row = 0; row < num_rows; row += 1) {
	if (!enable_pan)
		read_row = 0;
	tmp_ptr = int_ptr = (unsigned int FAR *) *((*cinfo->emethods->access_big_sarray)
			(raw_pic_ptr, read_row++, TRUE));
/*
	tmp_ptr2 = int_ptr2 = (unsigned int FAR *) *((*cinfo->emethods->access_big_sarray)
			(raw_pic_ptr, read_row++, TRUE));
*/
	inptr0 = input_data[0][row];
	inptr00 = input_data[0][row+1];
	inptr2 = input_data[2][row & 0xfffe];
	inptr1 = input_data[1][row & 0xfffe];
	for (col = 0; col < cols; col++) {
/* do row #1 */
		cb = GETJSAMPLE(inptr1[col]);
		cr = GETJSAMPLE(inptr2[col]);
		cg = cbtab[cr + (cb >> 1)];
		cb = Cbbtab[cb];
		cr = Crrtab[cr];

		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr0[col*2]);
		red = range_limit[cb];
		green = range_limit[cg];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		* int_ptr++ = (red >> 3) + ((green << 2) & 0x3e0) + ((blue << 7) & 0xfc00);

		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr0[col*2+1]);
		red = range_limit[cb];
		green = range_limit[cg];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		* int_ptr++ = (red >> 3) + ((green << 2) & 0x3e0) + ((blue << 7) & 0xfc00);

/* row #2
		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr00[col*2]);
		red = range_limit[cb];
		green = range_limit[cg];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		* int_ptr2++ = (red >> 3) + ((green << 2) & 0x3e0) + ((blue << 7) & 0xfc00);

		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr00[col*2+1]);
		red = range_limit[cb];
		green = range_limit[cg];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		* int_ptr2++ = (red >> 3) + ((green << 2) & 0x3e0) + ((blue << 7) & 0xfc00);
*/
		}
	draw_line(gr_row++, (unsigned char FAR *)tmp_ptr, line_buffer_ptr);
/*
	draw_line(gr_row++, (unsigned char FAR *)tmp_ptr2, line_buffer_ptr);
*/
	}
}




/*
 * Convert some rows of samples to the output colorspace.  ==> 16 bit
 */

METHODDEF void
ycc_rgb_16_convert (decompress_info_ptr cinfo, int num_rows, long num_cols,
		 JSAMPIMAGE input_data, JSAMPIMAGE output_data)
{
  register int cg, cb, cr;
  register JSAMPROW inptr0, inptr1, inptr2;
  register int col;
  register JSAMPLE * range_limit;
  register int * Crrtab = Cr_r_tab;
  register int * Cbbtab = Cb_b_tab;
  register int * cbtab = Cb_tab;
  int row;


for (row = 0; row < num_rows; row++) {
	if (!enable_pan)
		read_row = 0;
	tmp_ptr = int_ptr = (unsigned int FAR *) *((*cinfo->emethods->access_big_sarray)
			(raw_pic_ptr, read_row++, TRUE));
	inptr2 = input_data[2][row];
	inptr1 = input_data[1][row];
	inptr0 = input_data[0][row];
	for (col = 0; col < num_cols; col++) {
		cb = GETJSAMPLE(inptr1[col]);
		cr = GETJSAMPLE(inptr2[col]);
		cg = cbtab[cr + (cb >> 1)];
		cb = Cbbtab[cb];
		cr = Crrtab[cr];
		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr0[col]);
		red = range_limit[cb];
		green = range_limit[cg];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		* int_ptr++ = (red >> 3) + ((green << 3) & 0x7e0) + ((blue << 8) & 0xf800);
		}
	draw_line(gr_row, (unsigned char FAR *)tmp_ptr, line_buffer_ptr);
	gr_row++;
	}
}


METHODDEF void
ycc_rgb_fst_16_convert (decompress_info_ptr cinfo, int num_rows, long num_cols,
		 JSAMPIMAGE input_data, JSAMPIMAGE output_data)
{
  register int cb, cr, cg;
  register JSAMPROW inptr0, inptr1, inptr2, inptr00;
  register int col, cols;
  register JSAMPLE * range_limit;
  int * Crrtab = Cr_r_tab;
  int * Cbbtab = Cb_b_tab;
  int * cbtab = Cb_tab;
  int row;

cols = num_cols >> 1;

for (row = 0; row < num_rows; row += 1) {
	if (!enable_pan)
		read_row = 0;
	tmp_ptr = int_ptr = (unsigned int FAR *) *((*cinfo->emethods->access_big_sarray)
			(raw_pic_ptr, read_row++, TRUE));
/*	tmp_ptr2 = int_ptr2 = (unsigned int FAR *) *((*cinfo->emethods->access_big_sarray)
			(raw_pic_ptr, read_row++, TRUE));
*/
	inptr0 = input_data[0][row];
	inptr00 = input_data[0][row+1];
	inptr2 = input_data[2][row & 0xfffe];
	inptr1 = input_data[1][row & 0xfffe];
	for (col = 0; col < cols; col++) {
/* do row #1 */
		cb = GETJSAMPLE(inptr1[col]);
		cr = GETJSAMPLE(inptr2[col]);
		cg = cbtab[cr + (cb >> 1)];
		cb = Cbbtab[cb];
		cr = Crrtab[cr];

		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr0[col*2]);
		red = range_limit[cb];
		green = range_limit[cg];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		* int_ptr++ = (red >> 3) + ((green << 3) & 0x7e0) + ((blue << 8) & 0xf800);

		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr0[col*2+1]);
		red = range_limit[cb];
		green = range_limit[cg];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		* int_ptr++ = (red >> 3) + ((green << 3) & 0x7e0) + ((blue << 8) & 0xf800);

/* row #2
		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr00[col*2]);
		red = range_limit[cb];
		green = range_limit[cg];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		* int_ptr2++ = (red >> 3) + ((green << 3) & 0x7e0) + ((blue << 8) & 0xf800);

		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr00[col*2+1]);
		red = range_limit[cb];
		green = range_limit[cg];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		* int_ptr2++ = (red >> 3) + ((green << 3) & 0x7e0) + ((blue << 8) & 0xf800);
*/
		}
	draw_line(gr_row++, (unsigned char FAR *)tmp_ptr, line_buffer_ptr);
/*
	draw_line(gr_row++, (unsigned char FAR *)tmp_ptr2, line_buffer_ptr);
*/
	}
}


METHODDEF void
ycc_rgb_24_convert (decompress_info_ptr cinfo, int num_rows, long num_cols,
		 JSAMPIMAGE input_data, JSAMPIMAGE output_data)
{
  register int y, cb, cr, temp;
  register JSAMPROW inptr0, inptr1, inptr2;
  register JSAMPROW outptr;
  register int col;
  register JSAMPLE * range_limit;
  register int * Crrtab = Cr_r_tab;
  register int * Cbbtab = Cb_b_tab;
  register int * cbtab = Cb_tab;
  int row, flip;

flip = defaults & rgb_flip;

for (row = 0; row < num_rows; row++) {
	if (!enable_pan)
		read_row = 0;
	tmp_ptr = outptr = (unsigned int FAR *) *((*cinfo->emethods->access_big_sarray)
			(raw_pic_ptr, read_row++, TRUE));
	inptr2 = input_data[2][row];
	inptr1 = input_data[1][row];
	inptr0 = input_data[0][row];
	for (col = 0; col < num_cols; col++) {
		cb = GETJSAMPLE(inptr1[col]);
		cr = GETJSAMPLE(inptr2[col]);
		y = cbtab[cr + (cb >> 1)];
		cr = Crrtab[cr];
		cb = Cbbtab[cb];
		if (!flip){
			temp = cb;
			cb = cr;
			cr = temp;
			}
		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr0[col]);
		red = range_limit[cb];
		green = range_limit[y];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		outptr[0] = blue;
		outptr[1] = green;
		outptr[2] = red;
		outptr += 3;
		}
	draw_line(gr_row, (unsigned char FAR *) tmp_ptr, line_buffer_ptr);
	gr_row++;
	}
}



METHODDEF void
ycc_rgb_fst_24_convert (decompress_info_ptr cinfo, int num_rows, long num_cols,
		 JSAMPIMAGE input_data, JSAMPIMAGE output_data)
{
  register int y, cb, cr, temp;
  register JSAMPROW inptr0, inptr1, inptr2, inptr00;
  register JSAMPROW outptr, outptr2;
  register int col, cols;
  register JSAMPLE * range_limit;
  register int * Crrtab = Cr_r_tab;
  register int * Cbbtab = Cb_b_tab;
  register int * cbtab = Cb_tab;
  int row, flip;

flip = defaults & rgb_flip;
cols = num_cols >> 1;

for (row = 0; row < num_rows; row +=1) {
	if (!enable_pan)
		read_row = 0;
	tmp_ptr = outptr = (unsigned int FAR *) *((*cinfo->emethods->access_big_sarray)
			(raw_pic_ptr, read_row++, TRUE));
/*
	tmp_ptr2 = outptr2 = (unsigned int FAR *) *((*cinfo->emethods->access_big_sarray)
			(raw_pic_ptr, read_row++, TRUE));
if (tmp_ptr2 == NULL){
	sound(500);
	delay(100);
	sound(1000);
	delay(200);
	nosound();
	}
*/
	inptr2 = input_data[2][row & 0xfffe];
	inptr1 = input_data[1][row & 0xfffe];
	inptr0 = input_data[0][row];
	inptr00 = input_data[0][row+1];
	for (col = 0; col < cols; col++){
		cb = GETJSAMPLE(inptr1[col]);
		cr = GETJSAMPLE(inptr2[col]);
		y = cbtab[cr + (cb >> 1)];
		cr = Crrtab[cr];
		cb = Cbbtab[cb];
		if (!flip){
			temp = cb;
			cb = cr;
			cr = temp;
			}
		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr0[col << 1]);
		red = range_limit[cb];
		green = range_limit[y];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		outptr[0] = blue;
		outptr[1] = green;
		outptr[2] = red;

		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr0[(col << 1)+1]);
		red = range_limit[cb];
		green = range_limit[y];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		outptr[3] = blue;
		outptr[4] = green;
		outptr[5] = red;
		outptr += 6;

/* row #2
		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr00[col << 1]);
		red = range_limit[cb];
		green = range_limit[y];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		outptr2[0] = blue;
		outptr2[1] = green;
		outptr2[2] = red;

		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr00[(col << 1)+1]);
		red = range_limit[cb];
		green = range_limit[y];
		blue = range_limit[cr];
		if (tint_loaded) tint_graphics();
		outptr2[3] = blue;
		outptr2[4] = green;
		outptr2[5] = red;
		outptr2 += 6;
*/
		}
	draw_line(gr_row++, (unsigned char FAR *) tmp_ptr, line_buffer_ptr);
/*
	draw_line(gr_row++, (unsigned char FAR *)tmp_ptr2, line_buffer_ptr);
*/
	}
}

#endif /* small_viewer */


/*
 * Convert some rows of samples to the output colorspace.
 */

METHODDEF void
ycc_rgb_convert (decompress_info_ptr cinfo, int num_rows, long num_cols,
		 JSAMPIMAGE input_data, JSAMPIMAGE output_data)
{
#ifdef SIXTEEN_BIT_SAMPLES
  register INT32 y;
  register UINT16 cb, cr;
#else
  register int cg, cb, cr;
#endif
  register JSAMPROW inptr0, inptr1, inptr2;
  register JSAMPROW outptr0, outptr1, outptr2;
  register int col;
  /* copy these pointers into registers if possible */
  register JSAMPLE * range_limit;
  register int * Crrtab = Cr_r_tab;
  register int * Cbbtab = Cb_b_tab;
  register int * cbtab = Cb_tab;
/*
  register INT32 * Crgtab = Cr_g_tab;
  register INT32 * Cbgtab = Cb_g_tab;
*/
  int row;
  SHIFT_TEMPS

  for (row = 0; row < num_rows; row++) {
	 inptr0 = input_data[0][row];
	 inptr1 = input_data[1][row];
    inptr2 = input_data[2][row];
	 outptr0 = output_data[0][row];
	 outptr1 = output_data[1][row];
    outptr2 = output_data[2][row];
	 for (col = 0; col < num_cols; col++) {
		cb = GETJSAMPLE(inptr1[col]);
		cr = GETJSAMPLE(inptr2[col]);
		cg = cbtab[cr + (cb >> 1)];
		cb = Cbbtab[cb];
		cr = Crrtab[cr];
		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr0[col]);
		outptr0[col] = range_limit[cr];	/* red */
		outptr1[col] = range_limit[cg];
		outptr2[col] = range_limit[cb];	/* blue */
	 }
  }
}




/*
 * Convert some rows of samples to the output colorspace.
 *
 * use faster output routines assuming non-upsampled data
 */

METHODDEF void
ycc_rgb_fst_convert (decompress_info_ptr cinfo, int num_rows, long num_cols,
		 JSAMPIMAGE input_data, JSAMPIMAGE output_data)
{
  register int y, cb, cr;
  register JSAMPROW inptr0, inptr1, inptr2;
  register JSAMPROW outptr0, outptr1, outptr2;
  register int col;
  register JSAMPLE * range_limit;
  register int * Crrtab = Cr_r_tab;
  register int * Cbbtab = Cb_b_tab;
  register int * cbtab = Cb_tab;
  int row;

  for (row = 0; row < num_rows; row++) {
	 inptr0 = input_data[0][row];
	 inptr1 = input_data[1][row & 0xfffe];
	 inptr2 = input_data[2][row & 0xfffe];
	 outptr0 = output_data[0][row];
	 outptr1 = output_data[1][row];
	 outptr2 = output_data[2][row];
	 for (col = num_cols-1; col >= 0; col--) {
		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr0[col]);
		cb = GETJSAMPLE(inptr1[col >> 1]);
		cr = GETJSAMPLE(inptr2[col >> 1]);
		y = cbtab[cr + (cb >> 1)];
		cr = Crrtab[cr];
		cb = Cbbtab[cb];
		outptr0[col] = range_limit[cr];
		outptr1[col] = range_limit[y];
		outptr2[col] = range_limit[cb];

		col--;
		range_limit = cinfo->sample_range_limit + GETJSAMPLE(inptr0[col]);
		outptr0[col] = range_limit[cr];
		outptr1[col] = range_limit[y];
		outptr2[col] = range_limit[cb];
	 }
  }
}





/*
 * Finish up at the end of the file.
 */

METHODDEF void
ycc_rgb_term (decompress_info_ptr cinfo)
{
  /* no work (we let free_all release the workspace) */
}


/**************** Cases other than YCbCr -> RGB **************/


/*
 * Initialize for colorspace conversion.
 */

METHODDEF void
null_init (decompress_info_ptr cinfo)
/* colorout_init for cases where no setup is needed */
{
  /* no work needed */
}


/*
 * Color conversion for no colorspace change: just copy the data.
 */

METHODDEF void
null_convert (decompress_info_ptr cinfo, int num_rows, long num_cols,
			JSAMPIMAGE input_data, JSAMPIMAGE output_data)
{
ERREXIT(cinfo->emethods, "Unsupported viewer colorspace");

/* short ci;
  for (ci = 0; ci < cinfo->num_components; ci++) {
	 jcopy_sample_rows(input_data[ci], 0, output_data[ci], 0,
				num_rows, num_cols);
  }*/
}


/*
 * Color conversion for grayscale: just copy the data.
 * This also works for YCbCr/YIQ -> grayscale conversion, in which
 * we just copy the Y (luminance) component and ignore chrominance.
 */

METHODDEF void
grayscale_convert (decompress_info_ptr cinfo, int num_rows, long num_cols,
		   JSAMPIMAGE input_data, JSAMPIMAGE output_data)
{
  jcopy_sample_rows(input_data[0], 0, output_data[0], 0,
			 num_rows, num_cols);
}


/*
 * Finish up at the end of the file.
 */

METHODDEF void
null_term (decompress_info_ptr cinfo)
/* colorout_term for cases where no teardown is needed */
{
  /* no work needed */
}



/*
 * The method selection routine for output colorspace conversion.
 */

GLOBAL void
jseldcolor (decompress_info_ptr cinfo)
{
int ci, video_card_type;

video_card_type = video_cards[video_mode_used].resolution;

  /* Make sure num_components agrees with jpeg_color_space */
  switch (cinfo->jpeg_color_space) {
  case CS_GRAYSCALE:
    if (cinfo->num_components != 1)
		ERREXIT(cinfo->emethods, "Bogus JPEG colorspace");
    break;

  case CS_RGB:
  case CS_YCbCr:
  case CS_YIQ:
    if (cinfo->num_components != 3)
      ERREXIT(cinfo->emethods, "Bogus JPEG colorspace");
    break;

  case CS_CMYK:
    if (cinfo->num_components != 4)
      ERREXIT(cinfo->emethods, "Bogus JPEG colorspace");
	 break;

  default:
	 ERREXIT(cinfo->emethods, "Unsupported JPEG colorspace");
	 break;
  }

  /* Set color_out_comps and conversion method based on requested space. */
  /* Also clear the component_needed flags for any unused components, */
  /* so that earlier pipeline stages can avoid useless computation. */

  switch (cinfo->out_color_space) {
  case CS_GRAYSCALE:
	 cinfo->color_out_comps = 1;
	 if (cinfo->jpeg_color_space == CS_GRAYSCALE ||
	cinfo->jpeg_color_space == CS_YCbCr ||
	cinfo->jpeg_color_space == CS_YIQ) {
		cinfo->methods->color_convert = grayscale_convert;
		cinfo->methods->colorout_init = null_init;
		cinfo->methods->colorout_term = null_term;
		/* For color->grayscale conversion, only the Y (0) component is needed */
		for (ci = 1; ci < cinfo->num_components; ci++)
	cinfo->cur_comp_info[ci]->component_needed = FALSE;
	 } else
		ERREXIT(cinfo->emethods, "Unsupported color conversion request");
	 break;
  case CS_RGB:
	 cinfo->color_out_comps = 3;
	 if (cinfo->jpeg_color_space == CS_YCbCr) {
#ifndef small_viewer
		switch (video_card_type){
			case SVGA_16_bit:
				if ((more_defaults & high_jpeg_quality) || (jpeg_type != H2V2))
					cinfo->methods->color_convert = ycc_rgb_16_convert;
				else
					cinfo->methods->color_convert = ycc_rgb_fst_16_convert;
				break;
			case SVGA_15_bit:
				if ((more_defaults & high_jpeg_quality) || (jpeg_type != H2V2))
					cinfo->methods->color_convert = ycc_rgb_15_convert;
				else
					cinfo->methods->color_convert = ycc_rgb_fst_15_convert;
				break;
			case SVGA_24_bit:
				if ((more_defaults & high_jpeg_quality) || (jpeg_type != H2V2))
					cinfo->methods->color_convert = ycc_rgb_24_convert;
				else
					cinfo->methods->color_convert = ycc_rgb_fst_24_convert;
				break;
			default:
				if ((more_defaults & high_jpeg_quality) || (cinfo->two_pass_quantize) || (jpeg_type != H2V2))
					cinfo->methods->color_convert = ycc_rgb_convert;
				else
					cinfo->methods->color_convert = ycc_rgb_fst_convert;
			}
#else
		cinfo->methods->color_convert = ycc_rgb_convert;
#endif
		cinfo->methods->colorout_init = ycc_rgb_init;
		cinfo->methods->colorout_term = ycc_rgb_term;
	 } else if (cinfo->jpeg_color_space == CS_RGB) {
		cinfo->methods->color_convert = null_convert;
		cinfo->methods->colorout_init = null_init;
		cinfo->methods->colorout_term = null_term;
	 } else
		ERREXIT(cinfo->emethods, "Unsupported color conversion request");
	 break;

  default:
	 /* Permit null conversion from CMYK or YCbCr to same output space */
	 if (cinfo->out_color_space == cinfo->jpeg_color_space) {
		cinfo->color_out_comps = cinfo->num_components;
		cinfo->methods->color_convert = null_convert;
		cinfo->methods->colorout_init = null_init;
      cinfo->methods->colorout_term = null_term;
    } else			/* unsupported non-null conversion */
      ERREXIT(cinfo->emethods, "Unsupported color conversion request");
    break;
  }

  if (cinfo->quantize_colors)
    cinfo->final_out_comps = 1;	/* single colormapped output component */
  else
    cinfo->final_out_comps = cinfo->color_out_comps;
}
