// ========================================================================
//  TODB LIBRARY
//    keyset.h
//
//    KSet class, KeySet class
//
//    Version: see TODB.H file
//
//    Copyright 1993 Christian Thrien
//    All rights reserved
// ========================================================================

#ifndef _KEYSET_H
#define _KEYSET_H

#include "kspersis.h"
#include "ptodb.h"

class ErrReporter;

//-------------------------------------------------------------------------
// KSet class
//    Keeps an ordered set of KeyType
//-------------------------------------------------------------------------

enum KSetError
{
    KS_ALLOC
};

class KSet
{
public:
    KSet();

    ~KSet();

    KSet & operator = ( const KSet & os );
    operator KeyType * ();
    operator unsigned char * ();

    // -- inserts a key
    Boolean Insert( KeyType k );
    // -- deletes a key
    Boolean Remove( KeyType k );
    // -- reads a key
    KeyType Get( KeyNbr n ) const;
    // -- cardinal
    KeyNbr Card() const;
    // -- creates an empty set of length l
    void Create( KeyNbr l );
    // -- loads the set
    void Put( void * buf, KeyNbr length );
    // -- updates card_ value
    void SetCard( KeyNbr n );

    // -- navigation, a returned value of 0 marks the end
    KeyType First();
    KeyType Next();

    // error control
    static void SetErrOut( ErrReporter * er );

private:
    KeyType * set_;
    KeyNbr card_;
    KeyNbr curr_index_;

    KSet( const KSet & os );

    // error control
    static ErrReporter * ErrOut;
    void ErrorHandler( KSetError err ) const;
};

inline KSet::~KSet()
{
    delete [] set_;
}

inline KSet::operator KeyType * ()
{
    return set_;
}

inline KSet::operator unsigned char * ()
{
    return (unsigned char *)set_;
}

inline KeyNbr KSet::Card() const
{
    return card_;
}

inline void KSet::Put( void * buf, KeyNbr length )
{
    memcpy( set_, buf, length * sizeof( KeyType ) );
    card_ = length;
}

inline void KSet::SetCard( KeyNbr n )
{
    card_ = n;
}

inline KeyType KSet::Get( KeyNbr n ) const
{
    return set_[ n ];
}

// returns first key of the set
// if empty, returns 0
inline KeyType KSet::First()
{
    curr_index_ = 0;
    return Next();
}

// returns next key
// a returned value of 0 marks the end
inline KeyType KSet::Next()
{
    if( curr_index_ < Card() )
      return set_[ curr_index_++ ];
    else
      return 0;
}

//-------------------------------------------------------------------------
// KeySet class
//    Persistent key set (saved in .KEY file, indexed in .IDX file)
//-------------------------------------------------------------------------

// The class id value 1 is used and values between 2 and 10 are reserved
// for upcoming compressed key set and others specialized data structure.
// If you want to save your own objects in the keyset data base,
// use a class id value greater than 10

enum SetId
{
    KEYSET = 1	// uncompressed persistent key set
};

class KeySet : public KSPersistent
{
public:
    KeySet( ObjAdr oa, TODB & db );
    KeySet( ObjAdr oa, TODB & db, Boolean append );

    ~KeySet();

    KSet & set();
    void AddKey( KeyType k );
    void RemoveKey( KeyType k );
    KeyNbr Card() const;

private:
    KSet set_;
    Boolean append_;
    KeyNbr card_;

    void Read();
    void Write();

    KeySet( const KeySet & ks );
};

inline KeySet::~KeySet()
{
    if ( append_ == False ) SaveObject();
}

inline KSet & KeySet::set()
{
    return set_;
}

inline void KeySet::RemoveKey( KeyType k )
{
    if ( set_.Remove( k ) == True )
	ChangeObject();
}

inline KeyNbr KeySet::Card() const
{
    return set_.Card();
}

#endif // _KEYSET_H
