/*
 * File......: AUDIT.PRG
 * Author....: Martin Colloby
 * BBS.......: The Dark Knight Returns
 * Net/Node..: 050/069
 * User Name.: Martin Colloby
 * Date......: 18/4/93
 * Revision..: 1.0
 *
 * This is an original work by Martin Colloby and is placed in the public
 * domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */


/*  $DOC$
 *  $FUNCNAME$
 *      GT_AUDIT()
 *  $CATEGORY$
 *     Audit
 *  $ONELINER$
 *     Update an audit trail
 *  $SYNTAX$
 *      GT_Audit( cTable , cOperation , lSecond )
 *  $ARGUMENTS$
 *      cTable     - Table to be audited
 *      cOperation - Operation that has been performed
 *      lSecond    - Should a second index be maintained
 *  $RETURNS$
 *      NIL
 *  $DESCRIPTION$
 *      The current record in cTable is written to the audit table (in
 *      AUDIT\) together with the date, time and username of the current
 *      operator.
 *      If lSecond is .T., an index identified as LEFT( cTable , 7 ) + "1"
 *      is opened and maintained.  The normal index (cTable) will be on
 *      auditdate+audittime, and the second index will be on the normal key
 *      of cTable.
 *
 *      cTable must be SELECTed before calling.
 *  $EXAMPLES$
 *
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *      GT_LIB.CH
 *  $END$
 */

#include "GT_LIB.ch"

FUNCTION GT_Audit( cTable , cOperation , lSecond )
MEMVAR cUserName

/****************************************************************************
 Purpose - Audit current record in table cTable
 Returns - None
 Author  - M K Colloby
 Created - 3/3/92
******************************************************************************
 Parameters - cTable      - Table to be audited
              cOperation  - Operation performed
              lSecond     - If .T., use two indexes
 Locals     - aFields     - Array to hold fields in cTable
              nCount      - Counting variable
 PUBLICS    - cUserName   - Name of user performing operation
****************************************************************************/

LOCAL aFields  :={}
LOCAL aIndexes := { "audit\" + cTable }
LOCAL nCount   := 0

// Select the table
SELECT ( cTable )

// Copy the contents of the data table into the array
FOR nCount := 1 TO FCOUNT()
    AADD( aFields , FIELDGET( nCount ) )
NEXT

IF lSecond
    AADD( aIndexes , "audit\" + ALLTRIM( LEFT( cTable , 7 ) ) + "1" )
ENDIF

// Open the Audit table
GT_UseDbf( "audit\" + cTable , aIndexes , "audit" , .F. , .F. , .F. , 5 )

// Add a blank record
GT_AddRec( 5 )

// Update the key fields
REPLACE audit->auditdate WITH DATE()
REPLACE audit->audittime WITH TIME()
REPLACE audit->audituser WITH cUserName
REPLACE audit->auditops  WITH cOperation

// This code block stuffs the contents of the array a_fields into the current
// table, but places them four fields to the right of their ordinal position
// in the array - this gets them past the first four key fields
AEVAL( aFields , { |value, count| FIELDPUT( count + 4 , value ) } )

// Unlock the audit table record
UNLOCK

// Release the audit table
SELECT "audit"
USE

// Go back to the source table
SELECT ( cTable )

RETURN NIL
*
