/*
    File......: GT_Menu.prg
    Author....: Martin Bryant
    BBS.......: The Dark Knight Returns
    Net/Node..: 050/069
    User Name.: Martin Bryant
    Date......: 05/02/93
    Revision..: 1.1

    This is an original work by Martin Bryant and is placed
    in the public domain.

    Modification history:
    ---------------------

    Rev 1.0 05/02/93
    PD Revision.

    Rev 1.1 18/05/93
    Fix Esc error.
*/

/*  $DOC$
 *  $FUNCNAME$
 *      GT_MENU()
 *  $CATEGORY$
 *      General
 *  $ONELINER$
 *      A general purpose menu function.
 *  $SYNTAX$
 *      GT_Menu(<aOptions>,[<aValids>],[<nTop>],[<nLeft>], ;
 *              [<nStart>],[<cTitle>],[<cColour>],[<cBox>]) ;
 *          -> nOption
 *  $ARGUMENTS$
 *      <aOptions> is an array of arrays. Each sub-array
 *      contains the menu options to be displayed. Each sub
 *      array is displayed alongside the previous.
 *
 *      <aValids> is a parrallel array of logical values
 *      that specify the selectability of each option. This
 *      will default to all options being selectable.
 *
 *      <nTop> and <nLeft> are the fixed top left corner
 *      for the menu. If ommited the menu is centered on
 *      the screen.
 *
 *      <nStart> is the option to start on.
 *
 *      <cTitle> is the title for the menu.
 *
 *      <cColour> is the colour settings to use. If
 *      ommitted, the current colours are used.
 *
 *      <cBox> is a character string defining the box
 *      characters to use. It should be eleven characters
 *      long.
 *  $RETURNS$
 *      A numeric value equal to the option number selected.
 *  $DESCRIPTION$
 *      A general purpose menu function. Allows multiple
 *      horizontal and verticle bars. The menu is
 *      optionally boxed and titled.
 *  $EXAMPLES$
 *      // Simple menu with 3 options, all selectable
 *      // positioned at 05, 10. The Highlite will start
 *      // on position 2 and the title will be
 *      // "Security Menu". The colours are specified as
 *      // are the box lines.
 *
 *      nOption := GT_Menu( ;
 *          {{  '1. Backup ', ;
 *              '2. Restore', ;
 *              'X.Exit    '}} ;
 *          {{.T.,.T.,.T.}}, ;
 *          05, ;
 *          10, ;
 *          2, ;
 *          'Security Menu', ;
 *          'R/W,N/R,W,W,N/W', ;
 *          '͸Գ ')
 *  $SEEALSO$
 *      GT_CHOOSE()
 *  $INCLUDE$
 *
 *  $END$
 */

#include "GT_LIB.ch"
#include "achoice.ch"

FUNCTION GT_Menu(aOptions,aValids,nTop,nLeft,nStart, ;
    cTitle,cColour,cBox)

Local aColumns := {}
Local cSaveColour := SETCOLOR()
Local nBars := LEN(aOptions)
Local nBottom  := 0
Local nCount :=0
Local nHeight :=0
Local nKey :=0
Local nOption := 0
Local nRight := 0
Local nWidth := 0

Default aOptions to {}
Default aValids to AFILL(ARRAY(LEN(aOptions)),.T.)
Default nTop to -1
Default nLeft to -1
Default nStart to 1
Default cTitle to ''
Default cColour to SETCOLOR()
Default cBox to BOX_DS

//  Width and columns
aColumns := ARRAY(nBars)
FOR nCount := 1 TO nBars
    aColumns[nCount] := nWidth
    nWidth += LEN(aOptions[nCount][1])
NEXT

//  Left ?
IF nLeft < 0
    nLeft := ROUND((MAXCOL() - nWidth) / 2,0)
ENDIF

//  Adjust Columns
FOR nCount := 1 TO nBars
    aColumns[nCount] += nLeft
NEXT

//  Height
nHeight := LEN(aOptions[1])

//  Top ?
IF nTop <  0
    nTop := ROUND(((MAXROW() - nHeight) / 2) - 2,0) + 3
ENDIF

//  nBottom
nBottom := nTop + nHeight - 1

//  Right
nRight := nLeft + nWidth - 01

//  Clear area
SETCOLOR(cColour)

IF .NOT. EMPTY(cBox)

    GT_Window(nTop-3,nLeft-1,nBottom+1,nRight+1,cBox)

    @ nTop-1,nLeft-1 SAY SUBSTR(cBox,11,1) + ;
        REPLICATE(SUBSTR(cBox,2,1),nRight-nLeft+1) + ;
        SUBSTR(cBox,10,1)

    @ nTop-2, nLeft+2 SAY cTitle

ELSE

    SCROLL(nTop,nLeft,nBottom,nRight,0)

ENDIF

FOR nCount := 1 TO nBars
    // Display menu all nBars
    KEYBOARD CHR(K_LEFT)
    ACHOICE(nTop,aColumns[nCount],nBottom, ;
        aColumns[nCount]+LEN(aOptions[nCount][1])-1, ;
        aOptions[nCount],aValids[nCount],'Ach_Menu')

NEXT

nOption := ((nStart-1) % LEN(aOptions[1])) + 1
nStart := nOption
nCount := INT((nStart-1) / LEN(aOptions[1])) + 1
KEYBOARD CHR(0)

DO WHILE .T.

    nStart := nOption

    nOption := ACHOICE(nTop,aColumns[nCount],nBottom, ;
        aColumns[nCount]+LEN(aOptions[nCount][1])-1, ;
        aOptions[nCount],aValids[nCount],'Ach_Menu',nStart)

    nKey := LASTKEY()
    DO CASE

        CASE nKey = K_ESC
            // Esc to last option of last bar
            IF nCount = nBars .AND. ROW() = nBottom
                EXIT
            ELSE
                nCount := nBars
                nOption := LEN(aOptions[nBars])
                LOOP
            ENDIF

        CASE nKey = K_LEFT
            nCount --
            IF nCount < 1
                nCount := nBars
            ENDIF

        CASE nKey = K_RIGHT
            nCount ++
            IF nCount > nBars
                nCount := 1
            ENDIF

            OTHERWISE
            EXIT

    ENDCASE

    nOption := ROW() - nTop + 1

ENDDO

nOption := ROUND(ROW() - nTop + 1 + ((nCount - 1) * LEN(aOptions[1])),0)

SETCOLOR(cSaveColour)

/*
    End of GT_Menu()
*/
RETURN(nOption)

FUNCTION Ach_menu(nMode,nElement,nRow)

LOCAL nKey := LASTKEY()
LOCAL nReturn := AC_CONT

Default nMode to AC_NOITEM
Default nElement to 0
Default nRow to nElement

DO CASE
    CASE nMode = AC_IDLE
        // Tum de tum

    CASE nMode = AC_HITTOP
        // Goto bottom
        KEYBOARD CHR(K_PGDN)

    CASE nMode = AC_HITBOTTOM
        // Goto Top
        KEYBOARD CHR(K_PGUP)

    CASE nMode = AC_EXCEPT
        // Special Key
        DO CASE
            CASE nKey = K_RETURN
                // Select
                nReturn := AC_SELECT

            CASE nKey = K_ESC
                // Exit
                nReturn := AC_ABORT

            CASE ISALPHA(CHR(nKey)) .OR. ISDIGIT(CHR(nKey))
                // Goto
                nReturn := AC_GOTO

            CASE nKey = K_RIGHT
                // Exit (for multiple bar menus)
                nReturn := AC_ABORT

            CASE nKey = K_LEFT
                // Exit (for multiple bar menus)
                nReturn := AC_ABORT

            OTHERWISE
                // ?

        ENDCASE

    OTHERWISE
        // Argh!
        nReturn := AC_ABORT

ENDCASE

/*
    End of Ach_menu()
*/
RETURN(nReturn)
