/* memory.c -- Memory management
   Copyright (c) 1994 by Eberhard Mattes

This file is part of emx.

emx is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

emx is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emx; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

As special exception, emx.dll can be distributed without source code
unless it has been changed.  If you modify emx.dll, this exception
no longer applies and you must remove this paragraph from all source
files for emx.dll.  */


#define INCL_DOSSEMAPHORES
#include <os2emx.h>
#include "emxdll.h"

/* This is the size of emx.dll's private memory pool. 1 MByte should
   be sufficient. */

#define MEMORY_SIZE     0x100000

/* Base address of emx.dll's private memory pool. */

static void *memory_base;

/* Mutex semaphore for protecting the heap. */

static HMTX heap_mutex;


/* Allocate a memory object of SIZE bytes with allocation flags FLAGS.
   Return a pointer to the memory object.  Abort on error. */

void *allocate_obj (ULONG size, ULONG flags)
{
  ULONG rc;
  void *base;

  rc = DosAllocMem (&base, size, flags);
  if (rc != 0)
    {
      oprintf ("DosAllocMem failed, error code = 0x%.8x\r\n", rc);
      quit (255);
    }
  return (base);
}


/* Free the memory object MEM.  It must have been allocated by
   DosAllocMem or DosAllocSharedMem.  Return the OS/2 error code. */

ULONG free_obj (void *mem)
{
  return (DosFreeMem (mem));
}


/* Call DosSetMem.  If ERROR is -1, return for any error code.
   Otherwise, abort if the error code is nonzero and not equal to
   ERROR. */

ULONG setmem (ULONG base, ULONG size, ULONG flags, ULONG error)
{
  ULONG rc;

  if (debug_flags & DEBUG_SETMEM)
    oprintf ("Set memory: %.8x %.8x %.8x\r\n",
             (unsigned)base, (unsigned)size, (unsigned)flags);
  rc = DosSetMem ((void *)base, size, flags);
  if (rc != 0 && error != (ULONG)-1 && rc != error)
    {
      if (rc == 8)
        otext ("Out of swap space\r\n");
      else
        oprintf ("DosSetMem failed, error code = 0x%.8x\r\n", (unsigned)rc);
      quit (255);
    }
  return (rc);
}


/* Suballocate a block of memory of SIZE bytes from emx.dll's private
   memory pool.  Return the address of the block.  Abort on error. */

void *allocate_sub (ULONG size)
{
  ULONG rc;
  void *mem;

  rc = DosSubAllocMem (memory_base, &mem, size);
  if (rc != 0)
    {
      oprintf ("DosSubAllocMem failed, error code = 0x%.8x\r\n", (unsigned)rc);
      quit (255);
    }
  return (mem);
}


/* Suballocate a block of memory of SIZE bytes from emx.dll's private
   memory pool.  The address of the block is stored to *MEM.  Return
   the OS/2 error code. */

ULONG sub_alloc (void **mem, ULONG size)
{
  return (DosSubAllocMem (memory_base, mem, size));
}


/* Deallocate a block of memory.  MEM must have been returned by
   allocate_sub() or sub_alloc().  SIZE must match the size of the
   block, as passed to allocate_sub() or sub_alloc(). */

ULONG sub_free (void *mem, ULONG size)
{
  return (DosSubFreeMem (memory_base, mem, size));
}


/* Initialize emx.dll's private memory pool. */

void init_memory (void)
{
  ULONG rc;

  memory_base = allocate_obj (MEMORY_SIZE, PAG_READ | PAG_WRITE);
  rc = DosSubSetMem (memory_base,
                     DOSSUB_INIT | DOSSUB_SPARSE_OBJ | DOSSUB_SERIALIZE,
                     MEMORY_SIZE);
  if (rc != 0)
    {
      oprintf ("DosSubSetMem failed, error code = 0x%.8x\r\n", (unsigned)rc);
      quit (255);
    }
}


/* Initialize the heap of the application. */

void init_heap (void)
{
  create_mutex_sem (&heap_mutex);
}


/* Lock the heap and allocate the heap object if not already done.
   change_heap() must be called after start_heap() to unlock the
   heap. */

static void start_heap (void)
{
  request_mutex (heap_mutex);
  if (brk_ptr == 0)
    {
      heap_base = (ULONG)allocate_obj (heap_size, PAG_READ | PAG_WRITE);
      brk_ptr = heap_base;
      heap_end = heap_base + heap_size;
    }
}


/* Expand the heap by INCR bytes.  INCR is a positive number.  Return
   0 (success) or -1 (error).*/

static long expand_heap (ULONG incr)
{
  ULONG rc, new_brk, addr, size, rest;

  new_brk = brk_ptr + incr;
  if (new_brk < heap_base || new_brk > heap_end)
    return (-1);
  addr = brk_ptr;
  size = incr;
  if (addr & 0xfff)
    {
      rest = 0x1000 - (addr & 0xfff);
      if (rest >= size)
        size = 0;
      else
        {
          size -= rest;
          addr += rest;
        }
    }
  if (size != 0)
    {
      rc = setmem (addr, size, PAG_DEFAULT | PAG_COMMIT, (ULONG)-1);
      if (rc != 0)
        return (-1);
    }
  brk_ptr = new_brk;
  return (0);
}


/* Shrink the heap by DECR bytes.  DECR is a positive number.  Return
   0 (success) or -1 (error).*/

static long shrink_heap (ULONG decr)
{
  ULONG rc, new_brk, addr, high;

  new_brk = brk_ptr - decr;
  if (new_brk < heap_base || new_brk > heap_end)
    return (-1);
  addr = (new_brk + 0xfff) & ~0xfff;
  high = (brk_ptr + 0xfff) & ~0xfff;
  if (high > addr)
    {
      rc = setmem (addr, high - addr, PAG_DECOMMIT, (ULONG)-1);
      if (rc != 0)
        return (-1);
    }
  brk_ptr = new_brk;
  return (0);
}


/* Change the size of the heap and unlock the heap.  INCR is the value
   added to the current heap size.  Return 0 (success) or -1 (error).
   start_heap() must be called before change_heap() to lock the heap. */

static long change_heap (long incr)
{
  long result;

  if (incr > 0)
    result = expand_heap (incr);
  else if (incr < 0)
    result = shrink_heap (-incr);
  else
    result = 0;
  DosReleaseMutexSem (heap_mutex);
  return (result);
}


/* This function implements the __sbrk() system call. */

long do_sbrk (long incr)
{
  long result;

  start_heap ();
  result = brk_ptr;
  if (change_heap (incr) != 0)
    result = -1;
  return (result);
}


/* This function implements the __brk() system call. */

long do_brk (ULONG brkp)
{
  start_heap ();
  return (change_heap (brkp - brk_ptr));
}
