/******************************************************************* CLOCK.CC
 *									    *
 *		  Analog Clock for Presentation Manager 		    *
 *									    *
 *		  Original program by Charles Petzold			    *
 *									    *
 ****************************************************************************/

#define INCL_BASE
#define INCL_PM
#include <os2.h>

#include <process.h>
#include <stdarg.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "debug.h"
#include "support.h"

#include "about.h"
#include "config.h"
#include "process.h"
#include "profile.h"
#include "restring.h"

#include "clock.h"


/****************************************************************************
 *									    *
 *			 Definitions & Declarations			    *
 *									    *
 ****************************************************************************/

  // Constants

#define PROGRAM_NAME "CLOCK"
#define CLASS_NAME   PROGRAM_NAME

#define DATEFMT_MM_DD_YY    (0x0000)
#define DATEFMT_DD_MM_YY    (0x0001)
#define DATEFMT_YY_MM_DD    (0x0002)

enum { ALERT_TASKCOUNT, ALERT_LOAD } ;

#define WM_REFRESH WM_USER


  // Type Definitions

typedef struct
{
  SHORT cxClient ;
  SHORT cyClient ;
  SHORT cxPixelDiam ;
  SHORT cyPixelDiam ;
}
WINDOWINFO, *PWINDOWINFO ;

typedef struct	      // Parameters saved to system.
{
  SWP	 Position ;		// Window size & location.
  BOOL	 fPosition ;

  BOOL	 Hour24 ;		// User options.
  BOOL	 fHour24 ;

  BOOL	 HideControls ;
  BOOL	 fHideControls ;

  BOOL	 Chime ;
  BOOL	 fChime ;

  BOOL	 Float ;
  BOOL	 fFloat ;

  BOOL	 Analog ;
  BOOL	 fAnalog ;

  BOOL	 AlertType ;
  BOOL	 fAlertType ;

  USHORT AlertLevels [2] [2] ;
  BOOL	 fAlertLevels ;

  CHAR	 FontNameSize [80] ;	// Presentation Parameters
  BOOL	 fFontNameSize ;

  COLOR  BackColor ;
  BOOL	 fBackColor ;

  COLOR  TextColor ;
  BOOL	 fTextColor ;
}
PROFILE, *PPROFILE ;

typedef struct
{
  HAB		 Anchor ;
  HMODULE	 Library ;
  HINI           ProfileHandle ;

  PROFILE Profile ;

  HWND hwndTitleBar ;
  HWND hwndSysMenu ;
  HWND hwndMinMax ;

  DATETIME PreviousDateTime ;
  USHORT PreviousHour ;
  LONG xPixelsPerMeter, yPixelsPerMeter ;
  WINDOWINFO wi ;

  USHORT Alert ;

  ULONG MaxCount ;
  ULONG IdleCounter ;
  ULONG IdleCount ;
  TID	IdleLoopTID ;
  TID	MonitorLoopTID ;

  COUNTRYINFO CountryInfo ;
}
DATA, *PDATA ;

typedef struct
{
  HAB Anchor ;
  HMODULE Library ;
  HINI ProfileHandle ;
  BOOL Analog ;
  BOOL fAnalog ;
}
PARMS, *PPARMS ;

typedef struct
{
  volatile PULONG Counter ;
  HWND Owner ;
}
MONITOR_PARMS, *PMONITOR_PARMS ;


  // Function Prototypes

extern INT main ( int argc, PSZ argv[] ) ;

static MRESULT EXPENTRY MessageProcessor
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
) ;

static METHODFUNCTION Create ;
static METHODFUNCTION Destroy ;
static METHODFUNCTION Size ;
static METHODFUNCTION SaveApplication ;
static METHODFUNCTION Paint ;
static METHODFUNCTION Command ;
static METHODFUNCTION ResetDefaults ;
static METHODFUNCTION HideControlsCmd ;
static METHODFUNCTION Configure ;
static METHODFUNCTION About ;
static METHODFUNCTION ButtonDown ;
static METHODFUNCTION ButtonDblClick ;
static METHODFUNCTION PresParamChanged ;
static METHODFUNCTION SysColorChange ;
static METHODFUNCTION QueryKeysHelp ;
static METHODFUNCTION HelpError ;
static METHODFUNCTION ExtHelpUndefined ;
static METHODFUNCTION HelpSubitemNotFound ;
static METHODFUNCTION Refresh ;

static int GetProfile ( HINI ProfileHandle, PPROFILE Profile ) ;
static VOID PutProfile ( HINI ProfileHandle, PPROFILE Profile ) ;

static VOID RotatePoint ( POINTL aptl[],  SHORT sNum, SHORT sAngle ) ;
static VOID ScalePoint ( POINTL aptl[], SHORT sNum, PWINDOWINFO pwi ) ;
static VOID TranslatePoint ( POINTL aptl[], SHORT sNum, PWINDOWINFO pwi ) ;
static VOID DrawHand ( HPS hPS, POINTL aptlIn[], SHORT sNum, SHORT sAngle,
  PWINDOWINFO pwi ) ;
static void PaintBackground ( HWND hwnd, HPS hPS, PDATA Data, BOOL MustPaint ) ;
static void PaintBorder ( HWND hwnd, HPS hPS, PDATA Data, BOOL MustPaint ) ;
static void PaintDigitalTime ( HWND hwnd, HPS hPS, PDATA Data, PDATETIME DateTime ) ;

static void HideControls
(
  BOOL fHide,
  HWND hwndFrame,
  HWND hwndSysMenu,
  HWND hwndTitleBar,
  HWND hwndMinMax
) ;

static void MonitorLoopThread ( PMONITOR_PARMS Parms ) ;

static ULONG CalibrateLoadMeter ( void ) ;

static void CounterThread ( PULONG Counter ) ;

static HINI OpenProfile ( PSZ Name, HAB Anchor, HMODULE Library, HWND HelpInstance ) ;


/****************************************************************************
 *									    *
 *	Program Mainline						    *
 *									    *
 ****************************************************************************/

extern int main ( int argc, PSZ argv[] )
{
 /***************************************************************************
  * Initialize the process.						    *
  ***************************************************************************/

  Process Proc ;

 /***************************************************************************
  * Now WIN and GPI calls will work.  Open the language DLL.		    *
  ***************************************************************************/

  HMODULE Library = 0 ;

  if ( DosLoadModule ( NULL, 0, (PSZ)PROGRAM_NAME, &Library ) )
  {
    Debug ( HWND_DESKTOP, "ERROR: Unable to load " PROGRAM_NAME ".DLL." ) ;
    DosExit ( EXIT_PROCESS, 1 ) ;
  }

 /***************************************************************************
  * Get the program title.                        			    *
  ***************************************************************************/

  ResourceString Title ( Library, IDS_TITLE ) ;

 /***************************************************************************
  * Decipher command-line parameters.					    *
  ***************************************************************************/

  ResourceString ResetCommand ( Library, IDS_PARMS_RESET ) ;
  ResourceString AnalogCommand ( Library, IDS_PARMS_ANALOG ) ;
  ResourceString DigitalCommand ( Library, IDS_PARMS_DIGITAL ) ;

  BOOL Reset = FALSE ;
  BOOL Analog = TRUE ;
  BOOL fAnalog = FALSE ;

  while ( --argc )
  {
    argv ++ ;

    WinUpper ( Proc.QueryAnchor(), NULL, NULL, *argv ) ;

    if ( *argv[0] == '?' )
    {
      ResourceString Message ( Library, IDS_PARAMETERLIST ) ;

      WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message.Ptr(),
	Title.Ptr(), 0, MB_ENTER | MB_NOICON ) ;

      DosExit ( EXIT_PROCESS, 1 ) ;
    }

    if ( !strcmp ( PCHAR(*argv), PCHAR(ResetCommand.Ptr()) ) )
    {
      Reset = TRUE ;
      continue ;
    }

    if ( !strcmp ( PCHAR(*argv), PCHAR(AnalogCommand.Ptr()) ) )
    {
      Analog = TRUE ;
      fAnalog = TRUE ;
      continue ;
    }

    if ( !strcmp ( PCHAR(*argv), PCHAR(DigitalCommand.Ptr()) ) )
    {
      Analog = FALSE ;
      fAnalog = TRUE ;
      continue ;
    }

    {
      ResourceString Format ( Library, IDS_ERROR_INVALIDPARM ) ;

      BYTE Message [200] ;
      sprintf ( PCHAR(Message), PCHAR(Format.Ptr()), *argv ) ;

      WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message,
	Title.Ptr(), 0, MB_ENTER | MB_ICONEXCLAMATION ) ;

      DosExit ( EXIT_PROCESS, 1 ) ;
    }
  }

 /***************************************************************************
  * Create the help instance.						    *
  ***************************************************************************/

  HELPINIT HelpInit =
  {
    sizeof ( HELPINIT ),
    0L,
    NULL,
    MAKEP ( 0xFFFF, ID_MAIN ),
    0,
    0,
    0,
    0,
    NULL,
    CMIC_HIDE_PANEL_ID,
    (PSZ) ( PROGRAM_NAME ".HLP" )
  } ;

  ResourceString HelpTitle ( Library, IDS_HELPTITLE ) ;
  HelpInit.pszHelpWindowTitle = HelpTitle.Ptr() ;

  HWND HelpWindow = WinCreateHelpInstance ( Proc.QueryAnchor(), &HelpInit ) ;

  if ( HelpWindow == NULL )
  {
    ResourceString Message ( Library, IDS_ERROR_WINCREATEHELPINSTANCE ) ;

    WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message.Ptr(),
      Title.Ptr(), 0, MB_ENTER | MB_ICONEXCLAMATION ) ;
  }

 /***************************************************************************
  * Open/create the profile file.                           		    *
  ***************************************************************************/

  HINI ProfileHandle = OpenProfile ( PSZ(PROGRAM_NAME), Proc.QueryAnchor(), Library, HelpWindow ) ;

  if ( ProfileHandle == NULL )
  {
    ResourceString Message ( Library, IDS_ERROR_PRFOPENPROFILE ) ;
//  Log ( "%s\r\n", Message.Ptr() ) ;
    WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message.Ptr(),
      Title.Ptr(), 0, MB_ENTER | MB_ICONEXCLAMATION ) ;
    DosFreeModule ( Library ) ;
    DosExit ( EXIT_PROCESS, 1 ) ;
  }

 /***************************************************************************
  * If we're going to reset the program's profile, do it now.               *
  ***************************************************************************/

  if ( Reset )
  {
    PrfWriteProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, NULL, NULL, 0 ) ;
  }

 /***************************************************************************
  * Create the frame window.						    *
  ***************************************************************************/

  #pragma pack(2)
  struct
  {
    USHORT Filler ;
    USHORT cb ;
    ULONG  flCreateFlags ;
    USHORT hmodResources ;
    USHORT idResources ;
  }
  fcdata ;
  #pragma pack()

  fcdata.cb = sizeof(fcdata) - sizeof(fcdata.Filler) ;
  fcdata.flCreateFlags =
    FCF_TITLEBAR | FCF_SYSMENU | FCF_SIZEBORDER |
    FCF_MINMAX | FCF_NOBYTEALIGN | FCF_ACCELTABLE ;
  fcdata.hmodResources = 0 ;
  fcdata.idResources = ID_MAIN ;

  HWND FrameWindow = WinCreateWindow
  (
    HWND_DESKTOP,
    WC_FRAME,
    Title.Ptr(),
    0,
    0, 0, 0, 0,
    HWND_DESKTOP,
    HWND_TOP,
    ID_MAIN,
    &fcdata.cb,
    NULL
  ) ;

  if ( FrameWindow == NULL )
  {
    ResourceString Message ( Library, IDS_ERROR_WINCREATEFRAME ) ;

    WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message.Ptr(),
      Title.Ptr(), 0, MB_ENTER | MB_ICONEXCLAMATION ) ;

    PrfCloseProfile ( ProfileHandle ) ;
    DosFreeModule ( Library ) ;
    DosExit ( EXIT_PROCESS, 1 ) ;
  }

 /***************************************************************************
  * Associate the help instance with the frame window.			    *
  ***************************************************************************/

  if ( HelpWindow )
  {
    WinAssociateHelpInstance ( HelpWindow, FrameWindow ) ;
  }

 /***************************************************************************
  * Register the window class.						    *
  ***************************************************************************/

  if ( NOT WinRegisterClass ( Proc.QueryAnchor(), (PSZ)CLASS_NAME, MessageProcessor,
    CS_MOVENOTIFY | CS_SIZEREDRAW | CS_SYNCPAINT, sizeof(PVOID) ) )
  {
    ResourceString Format ( Library, IDS_ERROR_WINREGISTERCLASS ) ;

    BYTE Message [200] ;
    sprintf ( PCHAR(Message), PCHAR(Format.Ptr()), CLASS_NAME ) ;

    WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message,
      Title.Ptr(), 0, MB_ENTER | MB_ICONEXCLAMATION ) ;

    PrfCloseProfile ( ProfileHandle ) ;
    DosFreeModule ( Library ) ;
    DosExit ( EXIT_PROCESS, 1 ) ;
  }

 /***************************************************************************
  * Create client window.  If this fails, destroy frame and return.	    *
  ***************************************************************************/

  PARMS Parms ;
  Parms.Anchor = Proc.QueryAnchor() ;
  Parms.Library = Library ;
  Parms.ProfileHandle = ProfileHandle ;
  Parms.Analog = Analog ;
  Parms.fAnalog = fAnalog ;

  HWND ClientWindow = WinCreateWindow
  (
    FrameWindow,
    (PSZ)CLASS_NAME,
    (PSZ)"",
    0,
    0, 0, 0, 0,
    FrameWindow,
    HWND_BOTTOM,
    FID_CLIENT,
    &Parms,
    NULL
  ) ;

  if ( ClientWindow == NULL )
  {
    ResourceString Message ( Library, IDS_ERROR_WINCREATEWINDOW ) ;

    WinMessageBox ( HWND_DESKTOP, HWND_DESKTOP, Message.Ptr(),
      Title.Ptr(), 0, MB_ENTER | MB_ICONEXCLAMATION ) ;

    WinDestroyWindow ( FrameWindow ) ;
    if ( HelpWindow )
    {
      WinDestroyHelpInstance ( HelpWindow ) ;
    }
    PrfCloseProfile ( ProfileHandle ) ;
    DosFreeModule ( Library ) ;
    DosExit ( EXIT_PROCESS, 1 ) ;
  }

 /***************************************************************************
  * Wait for and process messages to the window's queue.  Terminate         *
  *   when the WM_QUIT message is received.				    *
  ***************************************************************************/

  QMSG QueueMessage ;
  while ( WinGetMsg ( Proc.QueryAnchor(), &QueueMessage, NULL, 0, 0 ) )
  {
    WinDispatchMsg ( Proc.QueryAnchor(), &QueueMessage ) ;
  }

 /***************************************************************************
  * Discard all that was requested of the system and terminate. 	    *
  ***************************************************************************/

  WinDestroyWindow ( FrameWindow ) ;

  if ( HelpWindow )
  {
    WinDestroyHelpInstance ( HelpWindow ) ;
  }

  PrfCloseProfile ( ProfileHandle ) ;

  DosFreeModule ( Library ) ;

  DosExit ( EXIT_PROCESS, 0 ) ;
}

/****************************************************************************
 *									    *
 *	Window Message Processor					    *
 *									    *
 ****************************************************************************/

static MRESULT EXPENTRY MessageProcessor
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Dispatch the message according to the method table and return the	    *
  *   result.  Any messages not defined above get handled by the system     *
  *   default window processor. 					    *
  ***************************************************************************/

  static METHOD Methods [] =
  {
    { WM_CREATE,		Create		    },
    { WM_DESTROY,		Destroy 	    },
    { WM_SIZE,			Size		    },
    { WM_MOVE,			Size		    },
    { WM_SAVEAPPLICATION,	SaveApplication     },
    { WM_PAINT, 		Paint		    },
    { WM_COMMAND,		Command 	    },
    { WM_BUTTON1DOWN,		ButtonDown	    },
    { WM_BUTTON2DOWN,		ButtonDown	    },
    { WM_BUTTON1DBLCLK, 	ButtonDblClick	    },
    { WM_BUTTON2DBLCLK, 	ButtonDblClick	    },
    { WM_PRESPARAMCHANGED,	PresParamChanged    },
    { WM_SYSCOLORCHANGE,	SysColorChange	    },
    { HM_QUERY_KEYS_HELP,	QueryKeysHelp	    },
    { HM_ERROR, 		HelpError	    },
    { HM_EXT_HELP_UNDEFINED,	ExtHelpUndefined    },
    { HM_HELPSUBITEM_NOT_FOUND, HelpSubitemNotFound },
    { WM_REFRESH,		Refresh 	    }
  } ;

  return ( DispatchMessage ( hwnd, msg, mp1, mp2, Methods, sizeof(Methods)/sizeof(Methods[0]), WinDefWindowProc ) ) ;
}

/****************************************************************************
 *									    *
 *	Initialize main window. 					    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Create
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Allocate instance data.						    *
  ***************************************************************************/

  PDATA Data = PDATA ( malloc ( sizeof(DATA) ) ) ;

  memset ( Data, 0, sizeof(DATA) ) ;

  WinSetWindowPtr ( hwnd, QWL_USER, Data ) ;

 /***************************************************************************
  * Grab any parameters from the WM_CREATE message.			    *
  ***************************************************************************/

  PPARMS Parms = PPARMS ( PVOIDFROMMP ( mp1 ) ) ;

  Data->Anchor = Parms->Anchor ;
  Data->Library = Parms->Library ;
  Data->ProfileHandle = Parms->ProfileHandle ;

 /***************************************************************************
  * Get profile data. Try the OS2.INI first, then try for private INI.      *
  *   If obtained from OS2.INI, erase it afterwards.                        *
  ***************************************************************************/

  if ( GetProfile ( HINI_USERPROFILE, &Data->Profile ) )
  {
    GetProfile ( Data->ProfileHandle, &Data->Profile ) ;
  }
  else
  {
    PrfWriteProfileData ( HINI_USERPROFILE, (PSZ)PROGRAM_NAME, NULL, NULL, 0 ) ;
  }

  if ( Parms->fAnalog )
  {
    Data->Profile.Analog = Parms->Analog ;
    Data->Profile.fAnalog = TRUE ;
  }

 /***************************************************************************
  * Get country information.						    *
  ***************************************************************************/

  {
    COUNTRYCODE CountryCode ;
    ULONG Count ;
    ULONG Status ;

    CountryCode.country = 0 ;
    CountryCode.codepage = 0 ;

    Status = DosGetCtryInfo ( sizeof(Data->CountryInfo), &CountryCode,
      &Data->CountryInfo, &Count ) ;
    if ( Status )
    {
      BYTE Message [80] ;
      WinLoadMessage ( Data->Anchor, Data->Library, IDS_ERROR_DOSGETCTRYINFO,
	sizeof(Message), Message ) ;
      Debug ( hwnd, PCHAR(Message), Status ) ;
      Data->CountryInfo.fsDateFmt = DATEFMT_MM_DD_YY ;
      Data->CountryInfo.fsTimeFmt = 0 ;
      Data->CountryInfo.szDateSeparator[0] = '/' ;
      Data->CountryInfo.szDateSeparator[1] = 0 ;
      Data->CountryInfo.szTimeSeparator[0] = ':' ;
      Data->CountryInfo.szTimeSeparator[1] = 0 ;
      Data->CountryInfo.szThousandsSeparator[0] = ',' ;
      Data->CountryInfo.szThousandsSeparator[1] = 0 ;
    }
  }

  if ( NOT Data->Profile.fHour24 )
  {
    Data->Profile.Hour24 = Data->CountryInfo.fsTimeFmt ;
  }

 /***************************************************************************
  * Get initial time.							    *
  ***************************************************************************/

  DosGetDateTime ( &Data->PreviousDateTime ) ;

  Data->PreviousHour = ( Data->PreviousDateTime.hours * 5 ) % 60
    + Data->PreviousDateTime.minutes/12 ;

 /***************************************************************************
  * Get the frame handle.						    *
  ***************************************************************************/

  HWND FrameWindow = WinQueryWindow ( hwnd, QW_PARENT ) ;

 /***************************************************************************
  * Get the control window handles.					    *
  ***************************************************************************/

  Data->hwndSysMenu  = WinWindowFromID ( FrameWindow, FID_SYSMENU  ) ;
  Data->hwndTitleBar = WinWindowFromID ( FrameWindow, FID_TITLEBAR ) ;
  Data->hwndMinMax   = WinWindowFromID ( FrameWindow, FID_MINMAX   ) ;

 /***************************************************************************
  * Add basic extensions to the system menu.				    *
  ***************************************************************************/

  static MENUITEM MenuSeparator =
    { MIT_END, MIS_SEPARATOR, 0, 0, NULL, 0 } ;

  AddSysMenuItem ( FrameWindow, &MenuSeparator, NULL ) ;

  static MENUITEM MenuItems [] =
  {
    { MIT_END, MIS_TEXT,      0, IDM_SAVE_APPLICATION, NULL, 0 },
    { MIT_END, MIS_TEXT,      0, IDM_RESET_DEFAULTS,   NULL, 0 },
    { MIT_END, MIS_TEXT,      0, IDM_HIDE_CONTROLS,    NULL, 0 },
    { MIT_END, MIS_TEXT,      0, IDM_CONFIGURE,        NULL, 0 },
  } ;

  for ( int i=0; i<sizeof(MenuItems)/sizeof(MenuItems[0]); i++ )
  {
    ResourceString MenuText ( Data->Library, i+IDS_SAVE_APPLICATION ) ;
    AddSysMenuItem ( FrameWindow, MenuItems+i, MenuText.Ptr() ) ;
  }

 /***************************************************************************
  * Add 'About' to the system menu.					    *
  ***************************************************************************/

  AddSysMenuItem ( FrameWindow, &MenuSeparator, NULL ) ;

  ResourceString AboutText ( Data->Library, IDS_ABOUT ) ;

  static MENUITEM MenuAbout =
    { MIT_END, MIS_TEXT, 0, IDM_ABOUT, NULL, 0 } ;

  AddSysMenuItem ( FrameWindow, &MenuAbout, AboutText.Ptr() ) ;

 /***************************************************************************
  * Add 'Help' to the system menu.					    *
  ***************************************************************************/

  ResourceString HelpText ( Data->Library, IDS_HELP ) ;

  static MENUITEM MenuHelp =
    { MIT_END, MIS_HELP, 0, 0, NULL, 0 } ;

  AddSysMenuItem ( FrameWindow, &MenuHelp, HelpText.Ptr() ) ;

 /***************************************************************************
  * Calibrate the old-style load meter, if the high resolution timer's      *
  *   available.							    *
  ***************************************************************************/

  Data->MaxCount = (ULONG) max ( 1L, CalibrateLoadMeter ( ) ) ;

 /***************************************************************************
  * Start the load meter.  Put it to sleep if we're not using it.           *
  ***************************************************************************/

  DosCreateThread ( &Data->IdleLoopTID, CounterThread, (ULONG)&Data->IdleCounter, 0, 4096 ) ;
  DosSetPrty ( PRTYS_THREAD, PRTYC_IDLETIME, PRTYD_MINIMUM, Data->IdleLoopTID ) ;
  DosSuspendThread ( Data->IdleLoopTID ) ;

  Data->IdleCount = 0 ;
  Data->IdleCounter = 0 ;

  if ( Data->Profile.AlertType == ALERT_LOAD )
  {
    DosResumeThread ( Data->IdleLoopTID ) ;
  }

  PMONITOR_PARMS MonitorParms = PMONITOR_PARMS ( malloc ( sizeof(*MonitorParms) ) ) ;
  MonitorParms->Counter = & Data->IdleCounter ;
  MonitorParms->Owner = hwnd ;
  DosCreateThread ( &Data->MonitorLoopTID, MonitorLoopThread, (ULONG)MonitorParms, 2, 8192 ) ;

 /***************************************************************************
  * Add the program to the system task list.				    *
  ***************************************************************************/

  ResourceString Title ( Data->Library, IDS_TITLE ) ;
  Add2TaskList ( FrameWindow, Title.Ptr() ) ;

 /***************************************************************************
  * Get device capabilities before you mess around with resizing.	    *
  ***************************************************************************/

  HPS hPS = WinGetPS ( hwnd ) ;
  HDC hDC = GpiQueryDevice ( hPS ) ;
  DevQueryCaps ( hDC, CAPS_VERTICAL_RESOLUTION, 1L,
    &Data->yPixelsPerMeter ) ;
  DevQueryCaps ( hDC, CAPS_HORIZONTAL_RESOLUTION, 1L,
    &Data->xPixelsPerMeter ) ;
  WinReleasePS ( hPS ) ;

 /***************************************************************************
  * If window hasn't been positioned before, set the default position.      *
  ***************************************************************************/

  RECTL Rectangle ;
  if ( NOT Data->Profile.fPosition )
  {
    WinQueryWindowRect ( HWND_DESKTOP, &Rectangle ) ;

    Rectangle.xLeft = Rectangle.xRight - Rectangle.xRight / 4 ;
    Rectangle.yBottom = Rectangle.yTop - Rectangle.yTop / 3 ;

    Data->Profile.Position.x = (SHORT) Rectangle.xLeft ;
    Data->Profile.Position.y = (SHORT) Rectangle.yBottom ;

    Data->Profile.Position.cx = (SHORT) ( Rectangle.xRight - Rectangle.xLeft ) ;
    Data->Profile.Position.cy = (SHORT) ( Rectangle.yTop - Rectangle.yBottom ) ;
  }

 /***************************************************************************
  * Position & size the window. 					    *
  ***************************************************************************/

  Rectangle.xLeft   = Data->Profile.Position.x ;
  Rectangle.xRight  = Data->Profile.Position.x + Data->Profile.Position.cx ;
  Rectangle.yBottom = Data->Profile.Position.y ;
  Rectangle.yTop    = Data->Profile.Position.y + Data->Profile.Position.cy ;

  if ( Data->Profile.HideControls )
  {
    WinSetParent ( Data->hwndSysMenu,  HWND_OBJECT, FALSE ) ;
    WinSetParent ( Data->hwndTitleBar, HWND_OBJECT, FALSE ) ;
    WinSetParent ( Data->hwndMinMax,   HWND_OBJECT, FALSE ) ;

    WinSendMsg ( FrameWindow, WM_UPDATEFRAME,
      MPFROMSHORT ( FCF_TITLEBAR | FCF_SYSMENU | FCF_MINBUTTON ), 0L ) ;

    WinCalcFrameRect ( FrameWindow, &Rectangle, TRUE ) ;

    WinSetParent ( Data->hwndSysMenu,  FrameWindow, TRUE ) ;
    WinSetParent ( Data->hwndTitleBar, FrameWindow, TRUE ) ;
    WinSetParent ( Data->hwndMinMax,   FrameWindow, TRUE ) ;

    WinSendMsg ( FrameWindow, WM_UPDATEFRAME,
      MPFROMSHORT ( FCF_TITLEBAR | FCF_SYSMENU | FCF_MINBUTTON ), 0L ) ;

    WinCalcFrameRect ( FrameWindow, &Rectangle, FALSE ) ;
  }

  WinSetWindowPos ( FrameWindow, HWND_BOTTOM,
    (SHORT) Rectangle.xLeft, (SHORT) Rectangle.yBottom,
    (SHORT) ( Rectangle.xRight - Rectangle.xLeft ),
    (SHORT) ( Rectangle.yTop - Rectangle.yBottom ),
    SWP_SIZE | SWP_MOVE | SWP_ZORDER |
    ( Data->Profile.Position.fl & SWP_MINIMIZE ) |
    ( Data->Profile.Position.fl & SWP_MAXIMIZE ) |
    ( Data->Profile.Position.fl & SWP_RESTORE ) ) ;

 /***************************************************************************
  * Hide the controls if so configured and not minimized.		    *
  ***************************************************************************/

  if ( Data->Profile.HideControls
    AND NOT ( Data->Profile.Position.fl & SWP_MINIMIZE ) )
  {
    CheckMenuItem ( FrameWindow, FID_SYSMENU, IDM_HIDE_CONTROLS, Data->Profile.HideControls ) ;

    HideControls
    (
      TRUE,
      FrameWindow,
      Data->hwndSysMenu,
      Data->hwndTitleBar,
      Data->hwndMinMax
    ) ;
  }

 /***************************************************************************
  * Get the saved presentation parameters and reinstate them.		    *
  ***************************************************************************/

  if ( Data->Profile.fFontNameSize )
  {
    WinSetPresParam ( hwnd, PP_FONTNAMESIZE,
      strlen(Data->Profile.FontNameSize)+1, Data->Profile.FontNameSize ) ;
  }

  if ( Data->Profile.fBackColor )
  {
    WinSetPresParam ( hwnd, PP_BACKGROUNDCOLOR,
      sizeof(Data->Profile.BackColor), &Data->Profile.BackColor ) ;
  }

  if ( Data->Profile.fTextColor )
  {
    WinSetPresParam ( hwnd, PP_FOREGROUNDCOLOR,
      sizeof(Data->Profile.TextColor), &Data->Profile.TextColor ) ;
  }

 /***************************************************************************
  * Now that the window's in order, make it visible.                        *
  ***************************************************************************/

  WinShowWindow ( FrameWindow, TRUE ) ;

 /***************************************************************************
  * Success?  Return no error.						    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *									    *
 *	Destroy main window.						    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Destroy
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Release the instance memory.					    *
  ***************************************************************************/

  free ( Data ) ;

 /***************************************************************************
  * We're done.                                                             *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *									    *
 *	Resize the main window. 					    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Size
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Find out the window's new position and size.                            *
  ***************************************************************************/

  HWND FrameWindow = WinQueryWindow ( hwnd, QW_PARENT ) ;

  SWP Position ;
  WinQueryWindowPos ( FrameWindow, &Position ) ;

  if ( NOT ( Position.fl & SWP_MINIMIZE )
    AND NOT ( Position.fl & SWP_MAXIMIZE ) )
  {
    Data->Profile.Position.x = Position.x ;
    Data->Profile.Position.y = Position.y ;

    Data->Profile.Position.cx = Position.cx ;
    Data->Profile.Position.cy = Position.cy ;
  }

 /***************************************************************************
  * If the message was a sizing notification, recompute scaling factors.    *
  ***************************************************************************/

  if ( msg == WM_SIZE )
  {
    Data->wi.cxClient = SHORT1FROMMP ( mp2 ) ;
    Data->wi.cyClient = SHORT2FROMMP ( mp2 ) ;

    SHORT sDiamMM = (SHORT) min ( Data->wi.cxClient*1000L/Data->xPixelsPerMeter,
      Data->wi.cyClient*1000L/Data->yPixelsPerMeter ) ;

    Data->wi.cxPixelDiam = (SHORT) ( Data->xPixelsPerMeter * sDiamMM / 1000 ) ;
    Data->wi.cyPixelDiam = (SHORT) ( Data->yPixelsPerMeter * sDiamMM / 1000 ) ;
  }

 /***************************************************************************
  * If hiding the controls . . .					    *
  ***************************************************************************/

  if ( Data->Profile.HideControls )
  {

   /*************************************************************************
    * If changing to or from minimized state . . .			    *
    *************************************************************************/

    if ( ( Position.fl & SWP_MINIMIZE ) != ( Data->Profile.Position.fl & SWP_MINIMIZE ) )
    {

     /***********************************************************************
      * Hide the controls if no longer minimized.			    *
      ***********************************************************************/

      HideControls
      (
	NOT ( Position.fl & SWP_MINIMIZE ),
	FrameWindow,
	Data->hwndSysMenu,
	Data->hwndTitleBar,
	Data->hwndMinMax
      ) ;
    }
  }

  Data->Profile.Position.fl = Position.fl ;

 /***************************************************************************
  * We're done.                                                             *
  ***************************************************************************/

  return ( 0 ) ;
}

/****************************************************************************
 *									    *
 *	Process SAVE APPLICATION message.				    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY SaveApplication
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Call function to put all profile data out to the system.		    *
  ***************************************************************************/

  PutProfile ( Data->ProfileHandle, &Data->Profile ) ;

 /***************************************************************************
  * We're done.  Let the system complete default processing.                *
  ***************************************************************************/

  return ( WinDefWindowProc ( hwnd, WM_SAVEAPPLICATION, 0, 0 ) ) ;
}

/****************************************************************************
 *									    *
 *	Process timer message.						    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Refresh
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  *				Declarations				    *
  ***************************************************************************/

  static POINTL
    aptlHour[5]   = { 0,-15,  10,0,  0,60,  -10,0,  0,-15 },
    aptlMinute[5] = { 0,-20,   5,0,  0,80,   -5,0,  0,-20 },
    aptlSecond[2] = { 0,  0,   0,80 } ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * If we're supposed to float the window, do so here.                      *
  ***************************************************************************/

  if ( Data->Profile.Float )
    WinSetWindowPos ( WinQueryWindow(hwnd,QW_PARENT), HWND_TOP, 0, 0, 0, 0, SWP_ZORDER ) ;

 /***************************************************************************
  * Save the idle counter.						    *
  ***************************************************************************/

  Data->IdleCount = LONGFROMMP ( mp1 ) ;

 /***************************************************************************
  * Get current time.							    *
  ***************************************************************************/

  DATETIME DateTime ;
  DosGetDateTime ( &DateTime ) ;

  USHORT Hour = ( ( DateTime.hours * 5 ) % 60 + DateTime.minutes / 12 ) ;

 /***************************************************************************
  * Get presentation space.						    *
  ***************************************************************************/

  HPS hPS = WinGetPS ( hwnd ) ;
  GpiCreateLogColorTable ( hPS, LCOL_RESET, LCOLF_RGB, 0L, 0L, NULL ) ;

 /***************************************************************************
  * If analog or minimized . . .					    *
  ***************************************************************************/

  if ( Data->Profile.Analog OR ( Data->Profile.Position.fl & SWP_MINIMIZE ) )
  {

   /*************************************************************************
    * If a new minute has arrived, and the clock is minimized, clear it.    *
    *************************************************************************/

    if ( ( Data->Profile.Position.fl & SWP_MINIMIZE )
      AND ( DateTime.minutes != Data->PreviousDateTime.minutes ) )
    {
      PaintBackground ( hwnd, hPS, Data, TRUE ) ;
    }

   /*************************************************************************
    * Erase the old clock hands.					    *
    *************************************************************************/

    GpiSetColor ( hPS, Data->Profile.BackColor ) ;
    if ( ( Data->wi.cxClient > 50 ) AND ( Data->wi.cyClient > 50 ) )
    {
      DrawHand ( hPS, aptlSecond, 2, Data->PreviousDateTime.seconds, &Data->wi ) ;
    }

    if ( ( Hour != Data->PreviousHour )
      OR ( DateTime.minutes != Data->PreviousDateTime.minutes ) )
    {
      DrawHand ( hPS, aptlHour,   5, Data->PreviousHour,	     &Data->wi ) ;
      DrawHand ( hPS, aptlMinute, 5, Data->PreviousDateTime.minutes, &Data->wi ) ;
    }

   /*************************************************************************
    * Draw the new clock hands. 					    *
    *************************************************************************/

    GpiSetColor ( hPS, Data->Profile.TextColor ) ;
    DrawHand ( hPS, aptlHour,	5, Hour,	     &Data->wi ) ;
    DrawHand ( hPS, aptlMinute, 5, DateTime.minutes, &Data->wi ) ;
    if ( ( Data->wi.cxClient > 50 ) AND ( Data->wi.cyClient > 50 ) )
    {
      DrawHand ( hPS, aptlSecond, 2, DateTime.seconds, &Data->wi ) ;
    }
  }

 /***************************************************************************
  * Else if digital . . .						    *
  ***************************************************************************/

  else
  {

   /*************************************************************************
    * If minute has changed . . .					    *
    *************************************************************************/

    if ( DateTime.minutes != Data->PreviousDateTime.minutes )
    {
      PaintDigitalTime ( hwnd, hPS, Data, &DateTime ) ;
    }
  }

 /***************************************************************************
  * Adjust the border color to suit the current alert level.		    *
  ***************************************************************************/

  PaintBorder ( hwnd, hPS, Data, FALSE ) ;

 /***************************************************************************
  * Release presentation space. 					    *
  ***************************************************************************/

  WinReleasePS ( hPS ) ;

 /***************************************************************************
  * Chime if we've passed the top of the hour.  Save time when done.        *
  ***************************************************************************/

  if ( Data->Profile.Chime AND ( DateTime.hours != Data->PreviousDateTime.hours ) )
  {
    DosBeep  ( 400, 100 ) ;
    DosSleep (	    100 ) ;
    DosBeep  ( 400, 100 ) ;
  }

  Data->PreviousDateTime = DateTime ;
  Data->PreviousHour = Hour ;

  return ( 0 ) ;
}

/****************************************************************************
 *									    *
 *	Repaint entire window.						    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Paint
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  *				Declarations				    *
  ***************************************************************************/

  static POINTL
    aptlHour[5]   = { 0,-15,  10,0,  0,60,  -10,0,  0,-15 },
    aptlMinute[5] = { 0,-20,   5,0,  0,80,   -5,0,  0,-20 },
    aptlSecond[2] = { 0,  0,   0,80 } ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Get presentation space and make it use RGB colors.			    *
  ***************************************************************************/

  HPS hPS = WinBeginPaint ( hwnd, NULL, NULL ) ;
  GpiCreateLogColorTable ( hPS, LCOL_RESET, LCOLF_RGB, 0L, 0L, NULL ) ;

 /***************************************************************************
  * Paint the background.						    *
  ***************************************************************************/

  PaintBackground ( hwnd, hPS, Data, TRUE ) ;

 /***************************************************************************
  * If analog or minimized . . .					    *
  ***************************************************************************/

  if ( Data->Profile.Analog OR ( Data->Profile.Position.fl & SWP_MINIMIZE ) )
  {

   /*************************************************************************
    * Draw hour, minute and second hands.				    *
    *************************************************************************/

    DrawHand ( hPS, aptlHour,	5, Data->PreviousHour,		   &Data->wi ) ;
    DrawHand ( hPS, aptlMinute, 5, Data->PreviousDateTime.minutes, &Data->wi ) ;

    if ( ( Data->wi.cxClient > 50 ) AND ( Data->wi.cyClient > 50 ) )
    {
      DrawHand ( hPS, aptlSecond, 2, Data->PreviousDateTime.seconds, &Data->wi ) ;
    }
  }

 /***************************************************************************
  * Else if digital . . .						    *
  ***************************************************************************/

  else
  {
    PaintDigitalTime ( hwnd, hPS, Data, &Data->PreviousDateTime ) ;
  }

 /***************************************************************************
  * Release presentation space and return.				    *
  ***************************************************************************/

  WinEndPaint ( hPS ) ;

  return ( 0 ) ;
}

/****************************************************************************
 *									    *
 *	Process commands received by Main Window			    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Command
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Dispatch the messages.  There is no default message processor.	    *
  ***************************************************************************/

  static METHOD Methods [] =
  {
    { IDM_SAVE_APPLICATION, SaveApplication },
    { IDM_RESET_DEFAULTS,   ResetDefaults   },
    { IDM_HIDE_CONTROLS,    HideControlsCmd },
    { IDM_CONFIGURE,	    Configure	    },
    { IDM_EXIT, 	    Exit	    },
    { IDM_ABOUT,	    About	    }
  } ;

  return ( DispatchMessage ( hwnd, SHORT1FROMMP(mp1), mp1, mp2, Methods, sizeof(Methods)/sizeof(Methods[0]), NULL ) ) ;
}

/****************************************************************************
 *									    *
 *	Process Reset Defaults menu command.				    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY ResetDefaults
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Reset all profile data for this program.				    *
  ***************************************************************************/

  PrfWriteProfileData ( Data->ProfileHandle, PSZ(PROGRAM_NAME), NULL, NULL, 0 ) ;

 /***************************************************************************
  * Reset the program's presentation parameters.                            *
  ***************************************************************************/

  WinRemovePresParam ( hwnd, PP_FONTNAMESIZE ) ;
  WinRemovePresParam ( hwnd, PP_FOREGROUNDCOLOR ) ;
  WinRemovePresParam ( hwnd, PP_BACKGROUNDCOLOR ) ;

 /***************************************************************************
  * Done.								    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *									    *
 *	Process Hide Controls menu command.				    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY HideControlsCmd
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Get the frame window handle.					    *
  ***************************************************************************/

  HWND FrameWindow = WinQueryWindow ( hwnd, QW_PARENT ) ;

 /***************************************************************************
  * Toggle the Hide Controls setting.					    *
  ***************************************************************************/

  Data->Profile.HideControls = Data->Profile.HideControls ? FALSE : TRUE ;
  Data->Profile.fHideControls = TRUE ;

 /***************************************************************************
  * If controls aren't hidden yet, update the menu check-mark.              *
  ***************************************************************************/

  if ( Data->Profile.HideControls )
    CheckMenuItem ( FrameWindow, FID_SYSMENU, IDM_HIDE_CONTROLS, Data->Profile.HideControls ) ;

 /***************************************************************************
  * If not minimized right now, hide or reveal the controls.		    *
  ***************************************************************************/

  if ( NOT ( Data->Profile.Position.fl & SWP_MINIMIZE ) )
  {
    HideControls
    (
      Data->Profile.HideControls,
      FrameWindow,
      Data->hwndSysMenu,
      Data->hwndTitleBar,
      Data->hwndMinMax
    ) ;
  }

 /***************************************************************************
  * If controls are no longer hidden, update the menu check-mark.	    *
  ***************************************************************************/

  if ( NOT Data->Profile.HideControls )
    CheckMenuItem ( FrameWindow, FID_SYSMENU, IDM_HIDE_CONTROLS, Data->Profile.HideControls ) ;

 /***************************************************************************
  * Done.								    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *									    *
 *	Process Configure command.					    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY Configure
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Invoke the Configure dialog.					    *
  ***************************************************************************/

  CONFIG_PARMS Parms ;
  Parms.id	     = IDD_CONFIGURE ;
  Parms.hwndHelp     = WinQueryHelpInstance ( hwnd ) ;
  Parms.Analog	     = Data->Profile.Analog ;
  Parms.Hour24	     = Data->Profile.Hour24 ;
  Parms.HideControls = Data->Profile.HideControls ;
  Parms.Chime	     = Data->Profile.Chime ;
  Parms.Float	     = Data->Profile.Float ;
  Parms.AlertType    = Data->Profile.AlertType ;

  memcpy ( Parms.AlertLevels, Data->Profile.AlertLevels, sizeof(Parms.AlertLevels) ) ;

  if ( WinDlgBox ( HWND_DESKTOP, hwnd, ConfigureProcessor,
    Data->Library, IDD_CONFIGURE, &Parms ) )
  {
    Data->Profile.fHour24 = TRUE ;
    if ( Data->Profile.Hour24 != Parms.Hour24 )
    {
      Data->Profile.Hour24 = Parms.Hour24 ;
      WinInvalidateRect ( hwnd, NULL, FALSE ) ;
    }

    Data->Profile.fChime = TRUE ;
    Data->Profile.Chime = Parms.Chime ;

    Data->Profile.fFloat = TRUE ;
    Data->Profile.Float = Parms.Float ;

    Data->Profile.fAlertType = TRUE ;
    if ( Data->Profile.AlertType != Parms.AlertType )
    {
      Data->Profile.AlertType = Parms.AlertType ;
      if ( Data->Profile.AlertType == ALERT_LOAD )
	DosResumeThread ( Data->IdleLoopTID ) ;
      else
	DosSuspendThread ( Data->IdleLoopTID ) ;
    }

    Data->Profile.fAnalog = TRUE ;
    if ( Data->Profile.Analog != Parms.Analog )
    {
      Data->Profile.Analog = Parms.Analog ;
      WinInvalidateRect ( hwnd, NULL, FALSE ) ;
    }

    Data->Profile.fHideControls = TRUE ;
    if ( Data->Profile.HideControls != Parms.HideControls )
    {
      HWND FrameWindow = WinQueryWindow ( hwnd, QW_PARENT ) ;
      Data->Profile.HideControls = Parms.HideControls ;
      if ( Data->Profile.HideControls )
	CheckMenuItem ( FrameWindow, FID_SYSMENU, IDM_HIDE_CONTROLS, Data->Profile.HideControls ) ;
      if ( NOT ( Data->Profile.Position.fl & SWP_MINIMIZE ) )
      {
	HideControls
	(
	  Data->Profile.HideControls,
	  FrameWindow,
	  Data->hwndSysMenu,
	  Data->hwndTitleBar,
	  Data->hwndMinMax
	) ;
      }
      if ( NOT Data->Profile.HideControls )
	CheckMenuItem ( FrameWindow, FID_SYSMENU, IDM_HIDE_CONTROLS, Data->Profile.HideControls ) ;
    }

    Data->Profile.fAlertLevels = TRUE ;
    memcpy ( Data->Profile.AlertLevels, Parms.AlertLevels, sizeof(Parms.AlertLevels) ) ;
  }

 /***************************************************************************
  * Done.								    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *									    *
 *	Process About menu command.					    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY About
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Invoke the About dialog.						    *
  ***************************************************************************/

  ABOUT_PARMS Parms ;
  Parms.id = IDD_ABOUT ;
  Parms.hwndHelp = WinQueryHelpInstance ( hwnd ) ;

  WinDlgBox ( HWND_DESKTOP, hwnd, AboutProcessor,
    Data->Library, IDD_ABOUT, &Parms ) ;

 /***************************************************************************
  * Done.								    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *									    *
 *	Process Mouse Button being pressed.				    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY ButtonDown
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Determine the new window position.					    *
  ***************************************************************************/

  TRACKINFO TrackInfo ;
  memset ( &TrackInfo, 0, sizeof(TrackInfo) ) ;

  TrackInfo.cxBorder = 1 ;
  TrackInfo.cyBorder = 1 ;
  TrackInfo.cxGrid = 1 ;
  TrackInfo.cyGrid = 1 ;
  TrackInfo.cxKeyboard = 8 ;
  TrackInfo.cyKeyboard = 8 ;

  HWND FrameWindow = WinQueryWindow ( hwnd, QW_PARENT ) ;

  SWP Position ;
  WinQueryWindowPos ( FrameWindow, &Position ) ;
  TrackInfo.rclTrack.xLeft   = Position.x ;
  TrackInfo.rclTrack.xRight  = Position.x + Position.cx ;
  TrackInfo.rclTrack.yBottom = Position.y ;
  TrackInfo.rclTrack.yTop    = Position.y + Position.cy ;

  WinQueryWindowPos ( HWND_DESKTOP, &Position ) ;
  TrackInfo.rclBoundary.xLeft   = Position.x ;
  TrackInfo.rclBoundary.xRight  = Position.x + Position.cx ;
  TrackInfo.rclBoundary.yBottom = Position.y ;
  TrackInfo.rclBoundary.yTop    = Position.y + Position.cy ;

  TrackInfo.ptlMinTrackSize.x = 0 ;
  TrackInfo.ptlMinTrackSize.y = 0 ;
  TrackInfo.ptlMaxTrackSize.x = Position.cx ;
  TrackInfo.ptlMaxTrackSize.y = Position.cy ;

  TrackInfo.fs = TF_MOVE | TF_STANDARD | TF_ALLINBOUNDARY ;

  if ( WinTrackRect ( HWND_DESKTOP, NULL, &TrackInfo ) )
  {
    WinSetWindowPos ( FrameWindow, NULL,
      (SHORT) TrackInfo.rclTrack.xLeft,
      (SHORT) TrackInfo.rclTrack.yBottom,
      0, 0, SWP_MOVE ) ;
  }

 /***************************************************************************
  * Return through the default processor, letting window activation	    *
  *   and other system functions occur. 				    *
  ***************************************************************************/

  return ( WinDefWindowProc ( hwnd, msg, mp1, mp2 ) ) ;
}

/****************************************************************************
 *									    *
 *	Process Mouse Button having been double-clicked.		    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY ButtonDblClick
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Send message to self to stop hiding the controls.			    *
  ***************************************************************************/

  WinPostMsg ( hwnd, WM_COMMAND,
    MPFROM2SHORT ( IDM_HIDE_CONTROLS, 0 ),
    MPFROM2SHORT ( CMDSRC_OTHER, TRUE ) ) ;

 /***************************************************************************
  * Return through the default processor, letting window activation	    *
  *   and other system functions occur. 				    *
  ***************************************************************************/

  return ( WinDefWindowProc ( hwnd, msg, mp1, mp2 ) ) ;
}

/****************************************************************************
 *									    *
 *	Process Presentation Parameter Changed notification.		    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY PresParamChanged
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Get the presentation parameter that changed.			    *
  ***************************************************************************/

  switch ( LONGFROMMP(mp1) )
  {

   /*************************************************************************
    * If font, note the fact that we now have a font to be saved as	    *
    *	part of the configuration.  Get the font metrics and resize	    *
    *	the window appropriately.					    *
    *************************************************************************/

    case PP_FONTNAMESIZE:
    {
      ULONG ppid ;
      if ( WinQueryPresParam ( hwnd, PP_FONTNAMESIZE, 0, &ppid,
	sizeof(Data->Profile.FontNameSize), &Data->Profile.FontNameSize,
	0 ) )
      {
	Data->Profile.fFontNameSize = TRUE ;
      }
      else
      {
	strcpy ( (PCHAR)Data->Profile.FontNameSize, "" ) ;
	Data->Profile.fFontNameSize = FALSE ;
	PrfWriteProfileData ( Data->ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"FontNameSize", NULL, 0 ) ;
      }
      WinInvalidateRect ( hwnd, NULL, TRUE ) ;
      break ;
    }

   /*************************************************************************
    * If background color, note the fact and repaint the window.	    *
    *************************************************************************/

    case PP_BACKGROUNDCOLOR:
    {
      ULONG ppid ;
      if ( WinQueryPresParam ( hwnd, PP_BACKGROUNDCOLOR, 0, &ppid,
	sizeof(Data->Profile.BackColor), &Data->Profile.BackColor, 0 ) )
      {
	Data->Profile.fBackColor = TRUE ;
      }
      else
      {
	Data->Profile.BackColor = WinQuerySysColor ( HWND_DESKTOP, SYSCLR_WINDOW, 0L ) ;
	Data->Profile.fBackColor = FALSE ;
	PrfWriteProfileData ( Data->ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"BackgroundColor", NULL, 0 ) ;
      }
      WinInvalidateRect ( hwnd, NULL, TRUE ) ;
      break ;
    }

   /*************************************************************************
    * If foreground color, note the fact and repaint the window.	    *
    *************************************************************************/

    case PP_FOREGROUNDCOLOR:
    {
      ULONG ppid ;
      if ( WinQueryPresParam ( hwnd, PP_FOREGROUNDCOLOR, 0, &ppid,
	sizeof(Data->Profile.TextColor), &Data->Profile.TextColor, 0 ) )
      {
	Data->Profile.fTextColor = TRUE ;
      }
      else
      {
	Data->Profile.TextColor = WinQuerySysColor ( HWND_DESKTOP, SYSCLR_OUTPUTTEXT, 0L ) ;
	Data->Profile.fTextColor = FALSE ;
	PrfWriteProfileData ( Data->ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"ForegroundColor", NULL, 0 ) ;
      }
      WinInvalidateRect ( hwnd, NULL, TRUE ) ;
      break ;
    }
  }

 /***************************************************************************
  * Return through the default processor, letting window activation	    *
  *   and other system functions occur. 				    *
  ***************************************************************************/

  return ( WinDefWindowProc ( hwnd, msg, mp1, mp2 ) ) ;
}

/****************************************************************************
 *									    *
 *	Process System Color Change notification.			    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY SysColorChange
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * If we aren't using custom colors, then query for the new defaults.      *
  ***************************************************************************/

  if ( NOT Data->Profile.fBackColor )
  {
    Data->Profile.BackColor = WinQuerySysColor ( HWND_DESKTOP, SYSCLR_WINDOW, 0L ) ;
  }

  if ( NOT Data->Profile.fTextColor )
  {
    Data->Profile.TextColor = WinQuerySysColor ( HWND_DESKTOP, SYSCLR_OUTPUTTEXT, 0L ) ;
  }

 /***************************************************************************
  * Return value must be NULL, according to the documentation.		    *
  ***************************************************************************/

  return ( MRFROMP ( NULL ) ) ;
}

/****************************************************************************
 *									    *
 *	Process Query for Keys Help resource id.			    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY QueryKeysHelp
(
  HWND hwnd,
  USHORT msg,
  MPARAM mp1,
  MPARAM mp2
)
{
 /***************************************************************************
  * Simply return the ID of the Keys Help panel.			    *
  ***************************************************************************/

  return ( (MRESULT) IDM_KEYS_HELP ) ;
}

/****************************************************************************
 *									    *
 *	Process Help Manager Error					    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY HelpError
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  static struct
  {
    ULONG Error ;
    USHORT StringId ;
  }
  HelpErrors [] =
  {
    { HMERR_NO_FRAME_WND_IN_CHAIN,     IDS_HMERR_NO_FRAME_WND_IN_CHAIN },
    { HMERR_INVALID_ASSOC_APP_WND,     IDS_HMERR_INVALID_ASSOC_APP_WND },
    { HMERR_INVALID_ASSOC_HELP_INST,   IDS_HMERR_INVALID_ASSOC_HELP_IN },
    { HMERR_INVALID_DESTROY_HELP_INST, IDS_HMERR_INVALID_DESTROY_HELP_ },
    { HMERR_NO_HELP_INST_IN_CHAIN,     IDS_HMERR_NO_HELP_INST_IN_CHAIN },
    { HMERR_INVALID_HELP_INSTANCE_HDL, IDS_HMERR_INVALID_HELP_INSTANCE },
    { HMERR_INVALID_QUERY_APP_WND,     IDS_HMERR_INVALID_QUERY_APP_WND },
    { HMERR_HELP_INST_CALLED_INVALID,  IDS_HMERR_HELP_INST_CALLED_INVA },
    { HMERR_HELPTABLE_UNDEFINE,        IDS_HMERR_HELPTABLE_UNDEFINE    },
    { HMERR_HELP_INSTANCE_UNDEFINE,    IDS_HMERR_HELP_INSTANCE_UNDEFIN },
    { HMERR_HELPITEM_NOT_FOUND,        IDS_HMERR_HELPITEM_NOT_FOUND    },
    { HMERR_INVALID_HELPSUBITEM_SIZE,  IDS_HMERR_INVALID_HELPSUBITEM_S },
    { HMERR_HELPSUBITEM_NOT_FOUND,     IDS_HMERR_HELPSUBITEM_NOT_FOUND },
    { HMERR_INDEX_NOT_FOUND,	       IDS_HMERR_INDEX_NOT_FOUND       },
    { HMERR_CONTENT_NOT_FOUND,	       IDS_HMERR_CONTENT_NOT_FOUND     },
    { HMERR_OPEN_LIB_FILE,	       IDS_HMERR_OPEN_LIB_FILE	       },
    { HMERR_READ_LIB_FILE,	       IDS_HMERR_READ_LIB_FILE	       },
    { HMERR_CLOSE_LIB_FILE,	       IDS_HMERR_CLOSE_LIB_FILE        },
    { HMERR_INVALID_LIB_FILE,	       IDS_HMERR_INVALID_LIB_FILE      },
    { HMERR_NO_MEMORY,		       IDS_HMERR_NO_MEMORY	       },
    { HMERR_ALLOCATE_SEGMENT,	       IDS_HMERR_ALLOCATE_SEGMENT      },
    { HMERR_FREE_MEMORY,	       IDS_HMERR_FREE_MEMORY	       },
    { HMERR_PANEL_NOT_FOUND,	       IDS_HMERR_PANEL_NOT_FOUND       },
    { HMERR_DATABASE_NOT_OPEN,	       IDS_HMERR_DATABASE_NOT_OPEN     },
    { 0,			       IDS_HMERR_UNKNOWN	       }
  } ;

  ULONG ErrorCode = (ULONG) LONGFROMMP ( mp1 ) ;

 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Find the error code in the message table.				    *
  ***************************************************************************/

  int Index = 0 ;
  while ( HelpErrors[Index].Error
    AND ( HelpErrors[Index].Error != ErrorCode ) )
  {
    Index ++ ;
  }

 /***************************************************************************
  * Get the message texts.						    *
  ***************************************************************************/

  ResourceString Title ( Data->Library, IDS_HMERR ) ;

  ResourceString Message ( Data->Library, HelpErrors[Index].StringId ) ;

 /***************************************************************************
  * Display the error message.						    *
  ***************************************************************************/

  WinMessageBox
  (
    HWND_DESKTOP,
    hwnd,
    Message.Ptr(),
    Title.Ptr(),
    0,
    MB_OK | MB_WARNING
  ) ;

 /***************************************************************************
  * Return zero, indicating that the message was processed.		    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *									    *
 *	Process "Extended Help Undefined" notification			    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY ExtHelpUndefined
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Get the message texts.						    *
  ***************************************************************************/

  ResourceString Title ( Data->Library, IDS_HMERR ) ;

  ResourceString Message ( Data->Library, IDS_HMERR_EXTHELPUNDEFINED ) ;

 /***************************************************************************
  * Display the error message.						    *
  ***************************************************************************/

  WinMessageBox
  (
    HWND_DESKTOP,
    hwnd,
    Message.Ptr(),
    Title.Ptr(),
    0,
    MB_OK | MB_WARNING
  ) ;

 /***************************************************************************
  * Return zero, indicating that the message was processed.		    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}

/****************************************************************************
 *									    *
 *	Process "Help Subitem Not Found" notification			    *
 *									    *
 ****************************************************************************/

static MRESULT APIENTRY HelpSubitemNotFound
( 
  HWND hwnd, 
  USHORT msg, 
  MPARAM mp1, 
  MPARAM mp2
)
{
 /***************************************************************************
  * Find the instance data.						    *
  ***************************************************************************/

  PDATA Data = (PDATA) WinQueryWindowPtr ( hwnd, QWL_USER ) ;

 /***************************************************************************
  * Get the title text. 						    *
  ***************************************************************************/

  ResourceString Title ( Data->Library, IDS_HMERR ) ;

 /***************************************************************************
  * Format the error message.						    *
  ***************************************************************************/

  USHORT Topic = (USHORT) SHORT1FROMMP ( mp2 ) ;
  USHORT Subtopic = (USHORT) SHORT2FROMMP ( mp2 ) ;

  ResourceString Frame	 ( Data->Library, IDS_HELPMODE_FRAME ) ;
  ResourceString Menu	 ( Data->Library, IDS_HELPMODE_MENU ) ;
  ResourceString Window  ( Data->Library, IDS_HELPMODE_WINDOW ) ;
  ResourceString Unknown ( Data->Library, IDS_HELPMODE_UNKNOWN ) ;

  PBYTE Mode ;
  switch ( SHORT1FROMMP ( mp1 ) )
  {
    case HLPM_FRAME:
      Mode = Frame.Ptr() ;
      break ;

    case HLPM_MENU:
      Mode = Menu.Ptr() ;
      break ;

    case HLPM_WINDOW:
      Mode = Window.Ptr() ;
      break ;

    default:
      Mode = Unknown.Ptr() ;
  }

  ResourceString Format ( Data->Library, IDS_HELPSUBITEMNOTFOUND ) ;

  BYTE Message [200] ;
  sprintf ( (PCHAR)Message, PCHAR(Format.Ptr()), Mode, Topic, Subtopic ) ;

 /***************************************************************************
  * Display the error message.						    *
  ***************************************************************************/

  WinMessageBox
  (
    HWND_DESKTOP,
    hwnd,
    Message,
    Title.Ptr(),
    0,
    MB_OK | MB_WARNING
  ) ;

 /***************************************************************************
  * Return zero, indicating that the message was processed.		    *
  ***************************************************************************/

  return ( MRFROMSHORT ( 0 ) ) ;
}


/****************************************************************************
 *									    *
 *			     Get Profile Data				    *
 *									    *
 ****************************************************************************/

static int GetProfile ( HINI ProfileHandle, PPROFILE Profile )
{
 /***************************************************************************
  * Get the window's current size and position.                             *
  ***************************************************************************/

  #pragma pack(2)
  typedef struct {
    USHORT Filler ;
    USHORT fs ;
    USHORT cy, cx, y, x ;
    HWND hwndInsertBehind ;
    HWND hwnd ;
  } OLDSWP ;
  #pragma pack()

  ULONG Size ;
  memset ( &Profile->Position, 0, sizeof(Profile->Position) ) ;
  Profile->fPosition = FALSE ;
  if ( PrfQueryProfileSize ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"Position", &Size ) )
  {
    if ( Size == sizeof(OLDSWP)-sizeof(USHORT) )
    {
      OLDSWP OldPosition ;
      if ( PrfQueryProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"Position", &OldPosition.fs, &Size ) )
      {
	Profile->Position.fl = OldPosition.fs ;
        Profile->Position.cy = OldPosition.cy ;
        Profile->Position.cx = OldPosition.cx ;
        Profile->Position.y = OldPosition.y ;
        Profile->Position.x = OldPosition.x ;
        Profile->Position.hwndInsertBehind = OldPosition.hwndInsertBehind ;
        Profile->Position.hwnd = OldPosition.hwnd ;
        Profile->fPosition = TRUE ;
      }
    }
    else if ( Size == sizeof(Profile->Position) )
    {
      if ( PrfQueryProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"Position", &Profile->Position, &Size ) )
      {
        Profile->fPosition = TRUE ;
      }
    }
  }

  if ( NOT Profile->fPosition )
  {
    if ( ProfileHandle == HINI_USERPROFILE )
    {
      return ( 1 ) ;
    }
  }

 /***************************************************************************
  * Get the program options.						    *
  ***************************************************************************/

  Profile->Hour24 = FALSE ;
  if
  (
    PrfQueryProfileSize ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"Hour24", &Size )
    AND
    ( ( Size == sizeof(Profile->Hour24) ) OR ( Size == sizeof(SHORT) ) )
    AND
    PrfQueryProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"Hour24", &Profile->Hour24, &Size )
  )
  {
    Profile->fHour24 = TRUE ;
  }

  Profile->HideControls = FALSE ;
  if
  (
    PrfQueryProfileSize ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"HideControls", &Size )
    AND
    ( ( Size == sizeof(Profile->HideControls) ) OR ( Size == sizeof(SHORT) ) )
    AND
    PrfQueryProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"HideControls", &Profile->HideControls, &Size )
  )
  {
    Profile->fHideControls = TRUE ;
  }

  Profile->Chime = FALSE ;
  if
  (
    PrfQueryProfileSize ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"Chime", &Size )
    AND
    ( ( Size == sizeof(Profile->Chime) ) OR ( Size == sizeof(SHORT) ) )
    AND
    PrfQueryProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"Chime", &Profile->Chime, &Size )
  )
  {
    Profile->fChime = TRUE ;
  }

  Profile->Float = FALSE ;
  if
  (
    PrfQueryProfileSize ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"Float", &Size )
    AND
    ( ( Size == sizeof(Profile->Float) ) OR ( Size == sizeof(SHORT) ) )
    AND
    PrfQueryProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"Float", &Profile->Float, &Size )
  )
  {
    Profile->fFloat = TRUE ;
  }

  Profile->Analog = TRUE ;
  if
  (
    PrfQueryProfileSize ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"Analog", &Size )
    AND
    ( ( Size == sizeof(Profile->Analog) ) OR ( Size == sizeof(SHORT) ) )
    AND
    PrfQueryProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"Analog", &Profile->Analog, &Size )
  )
  {
    Profile->fAnalog = TRUE ;
  }

  Profile->AlertType = ALERT_TASKCOUNT ;
  if
  (
    PrfQueryProfileSize ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"AlertType", &Size )
    AND
    ( ( Size == sizeof(Profile->AlertType) ) OR ( Size == sizeof(SHORT) ) )
    AND
    PrfQueryProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"AlertType", &Profile->AlertType, &Size )
  )
  {
    Profile->fAlertType = TRUE ;
  }

  Profile->AlertLevels [ALERT_TASKCOUNT] [0] = 8 ;
  Profile->AlertLevels [ALERT_TASKCOUNT] [1] = 12 ;
  Profile->AlertLevels [ALERT_LOAD]	 [0] = 33 ;
  Profile->AlertLevels [ALERT_LOAD]	 [1] = 67 ;

  if
  (
    PrfQueryProfileSize ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"AlertLevels", &Size )
    AND
    ( Size == sizeof(Profile->AlertLevels) )
    AND
    PrfQueryProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"AlertLevels", Profile->AlertLevels, &Size )
  )
  {
    Profile->fAlertLevels = TRUE ;
  }

 /***************************************************************************
  * Get the presentation parameters.					    *
  ***************************************************************************/

  strcpy ( (PCHAR)Profile->FontNameSize, "" ) ;
  Profile->fFontNameSize = FALSE ;
  if
  (
    PrfQueryProfileSize ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"FontNameSize", &Size )
    AND
    ( Size == sizeof(Profile->FontNameSize) )
    AND
    PrfQueryProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"FontNameSize", &Profile->FontNameSize, &Size )
  )
  {
    Profile->fFontNameSize = TRUE ;
  }

  Profile->BackColor = WinQuerySysColor ( HWND_DESKTOP, SYSCLR_WINDOW, 0L ) ;
  Profile->fBackColor = FALSE ;
  if
  (
    PrfQueryProfileSize ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"BackgroundColor", &Size )
    AND
    ( Size == sizeof(Profile->BackColor) )
    AND
    PrfQueryProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"BackgroundColor", &Profile->BackColor, &Size )
  )
  {
    Profile->fBackColor = TRUE ;
  }

  Profile->TextColor = WinQuerySysColor ( HWND_DESKTOP, SYSCLR_OUTPUTTEXT, 0L ) ;
  Profile->fTextColor = FALSE ;
  if
  (
    PrfQueryProfileSize ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"ForegroundColor", &Size )
    AND
    ( Size == sizeof(Profile->TextColor) )
    AND
    PrfQueryProfileData ( ProfileHandle, (PSZ)PROGRAM_NAME, (PSZ)"ForegroundColor", &Profile->TextColor, &Size )
  )
  {
    Profile->fTextColor = TRUE ;
  }

 /***************************************************************************
  * Return no error.							    *
  ***************************************************************************/

  return ( 0 ) ;
}

/****************************************************************************
 *									    *
 *			     Put Profile Data				    *
 *									    *
 ****************************************************************************/

static VOID PutProfile ( HINI ProfileHandle, PPROFILE Profile )
{
 /***************************************************************************
  * Save the window's current size and position.                            *
  ***************************************************************************/

  PrfWriteProfileData
  (
    ProfileHandle,
    (PSZ)PROGRAM_NAME,
    (PSZ)"Position",
    &Profile->Position,
    sizeof(Profile->Position)
  ) ;

 /***************************************************************************
  * Save the program options.						    *
  ***************************************************************************/

  if ( Profile->fHour24 )
  {
    PrfWriteProfileData
    (
      ProfileHandle,
      (PSZ)PROGRAM_NAME,
      (PSZ)"Hour24",
      &Profile->Hour24,
      (ULONG)sizeof(Profile->Hour24)
    ) ;
  }

  if ( Profile->fHideControls )
  {
    PrfWriteProfileData
    (
      ProfileHandle,
      (PSZ)PROGRAM_NAME,
      (PSZ)"HideControls",
      &Profile->HideControls,
      (ULONG)sizeof(Profile->HideControls)
    ) ;
  }

  if ( Profile->fChime )
  {
    PrfWriteProfileData
    (
      ProfileHandle,
      (PSZ)PROGRAM_NAME,
      (PSZ)"Chime",
      &Profile->Chime,
      (ULONG)sizeof(Profile->Chime)
    ) ;
  }

  if ( Profile->fFloat )
  {
    PrfWriteProfileData
    (
      ProfileHandle,
      (PSZ)PROGRAM_NAME,
      (PSZ)"Float",
      &Profile->Float,
      (ULONG)sizeof(Profile->Float)
    ) ;
  }

  if ( Profile->fAnalog )
  {
    PrfWriteProfileData
    (
      ProfileHandle,
      (PSZ)PROGRAM_NAME,
      (PSZ)"Analog",
      &Profile->Analog,
      (ULONG)sizeof(Profile->Analog)
    ) ;
  }

  if ( Profile->fAlertType )
  {
    PrfWriteProfileData
    (
      ProfileHandle,
      (PSZ)PROGRAM_NAME,
      (PSZ)"AlertType",
      &Profile->AlertType,
      (ULONG)sizeof(Profile->AlertType)
    ) ;
  }

  if ( Profile->fAlertLevels )
  {
    PrfWriteProfileData
    (
      ProfileHandle,
      (PSZ)PROGRAM_NAME,
      (PSZ)"AlertLevels",
      Profile->AlertLevels,
      (ULONG)sizeof(Profile->AlertLevels)
    ) ;
  }

 /***************************************************************************
  * Save the presentation parameters.					    *
  ***************************************************************************/

  if ( Profile->fFontNameSize )
  {
    PrfWriteProfileData
    (
      ProfileHandle,
      (PSZ)PROGRAM_NAME,
      (PSZ)"FontNameSize",
      Profile->FontNameSize,
      sizeof(Profile->FontNameSize)
    ) ;
  }

  if ( Profile->fBackColor )
  {
    PrfWriteProfileData
    (
      ProfileHandle,
      (PSZ)PROGRAM_NAME,
      (PSZ)"BackgroundColor",
      &Profile->BackColor,
      sizeof(Profile->BackColor)
    ) ;
  }

  if ( Profile->fTextColor )
  {
    PrfWriteProfileData
    (
      ProfileHandle,
      (PSZ)PROGRAM_NAME,
      (PSZ)"ForegroundColor",
      &Profile->TextColor,
      sizeof(Profile->TextColor)
    ) ;
  }
}

/****************************************************************************
 *									    *
 *	Perform point coordinate rotation.				    *
 *									    *
 ****************************************************************************/

static VOID RotatePoint ( POINTL aptl[],  SHORT sNum, SHORT sAngle )
{
  static SHORT sSin[60] = {
      0,  105,  208,  309,  407,  500,  588,  669,  743,  809,
    866,  914,  951,  978,  995, 1000,  995,  978,  951,  914,
    866,  809,  743,  669,  588,  500,  407,  309,  208,  105,
      0, -104, -207, -308, -406, -499, -587, -668, -742, -808,
   -865, -913, -950, -977, -994, -999, -994, -977, -950, -913,
   -865, -808, -742, -668, -587, -499, -406, -308, -207, -104 };

  POINTL ptlTemp ;
  SHORT sIndex ;

  for ( sIndex = 0; sIndex < sNum ; sIndex++ )
  {
    ptlTemp.x = (aptl[sIndex].x * sSin[(sAngle+15)%60] +
      aptl[sIndex].y * sSin[sAngle]) / 1000 ;
    ptlTemp.y = (aptl[sIndex].y * sSin[(sAngle+15)%60] -
      aptl[sIndex].x * sSin[sAngle]) / 1000 ;
    aptl[sIndex] = ptlTemp ;
  }
}

/****************************************************************************
 *									    *
 *	Perform point coordinate scaling.				    *
 *									    *
 ****************************************************************************/

static VOID ScalePoint ( POINTL aptl[], SHORT sNum, PWINDOWINFO pwi )
{
  SHORT sIndex ;

  for ( sIndex=0; sIndex < sNum; sIndex++ )
  {
    aptl[sIndex].x = aptl[sIndex].x * pwi->cxPixelDiam / 200 ;
    aptl[sIndex].y = aptl[sIndex].y * pwi->cyPixelDiam / 200 ;
  }
}

/****************************************************************************
 *									    *
 *	Perform point coordinate translation.				    *
 *									    *
 ****************************************************************************/

static VOID TranslatePoint ( POINTL aptl[], SHORT sNum, PWINDOWINFO pwi )
{
  SHORT sIndex ;

  for ( sIndex=0; sIndex<sNum; sIndex++ )
  {
    aptl[sIndex].x += pwi->cxClient / 2 - 1 ;
    aptl[sIndex].y += pwi->cyClient / 2 - 1 ;
  }
}

/****************************************************************************
 *									    *
 *	Draw a clock hand.						    *
 *									    *
 ****************************************************************************/

static VOID DrawHand ( HPS hPS, POINTL aptlIn[], SHORT sNum, SHORT sAngle,
  PWINDOWINFO pwi )
{
  POINTL aptl[5] ;
  SHORT sIndex ;

  for ( sIndex=0; sIndex<sNum; sIndex++ )
  {
    aptl [ sIndex ] = aptlIn [ sIndex ] ;
  }

  RotatePoint ( aptl, sNum, sAngle ) ;
  ScalePoint ( aptl, sNum, pwi ) ;
  TranslatePoint ( aptl, sNum, pwi ) ;

  GpiMove ( hPS, aptl ) ;
  GpiPolyLine ( hPS, sNum-1L, aptl+1 ) ;
}

/****************************************************************************
 *									    *
 *		     Paint Background (all but hands)			    *
 *									    *
 ****************************************************************************/

static void PaintBackground ( HWND hwnd, HPS hPS, PDATA Data, BOOL MustPaint )
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  SHORT  Angle ;
  POINTL Points [3] ;
  RECTL  Rectangle ;

 /***************************************************************************
  * Clear the window.							    *
  ***************************************************************************/

  WinQueryWindowRect ( hwnd, &Rectangle ) ;

  GpiMove ( hPS, (PPOINTL) &Rectangle.xLeft ) ;
  GpiSetColor ( hPS, Data->Profile.BackColor ) ;
  GpiBox ( hPS, DRO_FILL, (PPOINTL) &Rectangle.xRight, 0L, 0L ) ;

 /***************************************************************************
  * Paint the appropriate border color. 				    *
  ***************************************************************************/

  PaintBorder ( hwnd, hPS, Data, MustPaint ) ;

 /***************************************************************************
  * If analog or minimized . . .					    *
  ***************************************************************************/

  if ( Data->Profile.Analog OR ( Data->Profile.Position.fl & SWP_MINIMIZE ) )
  {

   /*************************************************************************
    * Draw hour and minute marks around the dial.			    *
    *************************************************************************/

    GpiSetColor ( hPS, Data->Profile.TextColor ) ;

    for ( Angle=0; Angle<60; Angle++ )
    {
      Points[0].x = 0 ;
      Points[0].y = 90 ;
      RotatePoint ( Points, 1, Angle ) ;
      ScalePoint ( Points, 1, &Data->wi ) ;
      TranslatePoint ( Points, 1, &Data->wi ) ;
      Points[2].x = Points[2].y = Angle % 5 ? 2 : 10 ;

      ScalePoint ( Points + 2, 1, &Data->wi ) ;

      Points[0].x -= Points[2].x / 2 ;
      Points[0].y -= Points[2].y / 2 ;

      Points[1].x = Points[0].x + Points[2].x ;
      Points[1].y = Points[0].y + Points[2].y ;

      GpiMove ( hPS, Points ) ;

      if ( ( Angle % 5 == 0 ) OR
	( ( Data->wi.cxClient ) > 50 ) AND ( Data->wi.cyClient > 50 ) )
      {
	GpiBox ( hPS, DRO_OUTLINEFILL, Points+1, Points[2].x, Points[2].y ) ;
      }
    }
  }
}

/****************************************************************************
 *									    *
 *			      Paint Border				    *
 *									    *
 ****************************************************************************/

static void PaintBorder ( HWND hwnd, HPS hPS, PDATA Data, BOOL MustPaint )
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  USHORT Alert ;
  USHORT Level ;
  POINTL Point ;
  RECTL  Rectangle ;

 /***************************************************************************
  * Determine level of the current alert type.				    *
  ***************************************************************************/

  if ( Data->Profile.AlertType == ALERT_TASKCOUNT )
  {
    Level = WinQuerySwitchList ( Data->Anchor, NULL, 0 ) ;
  }
  else
  {
    Data->MaxCount = (ULONG) max ( Data->MaxCount, Data->IdleCount ) ;
    Level = (USHORT) ( ( ( Data->MaxCount - Data->IdleCount ) * 100 ) / Data->MaxCount ) ;
  }

 /***************************************************************************
  * Determine alert class.						    *
  ***************************************************************************/

  if ( Level < Data->Profile.AlertLevels[Data->Profile.AlertType][0] )
    Alert = 0 ;
  else if ( Level < Data->Profile.AlertLevels[Data->Profile.AlertType][1] )
    Alert = 1 ;
  else
    Alert = 2 ;

 /***************************************************************************
  * If border must be painted, or the alert level has changed, paint it.    *
  ***************************************************************************/

  if ( MustPaint OR ( Alert != Data->Alert ) )
  {
    WinQueryWindowRect ( hwnd, &Rectangle ) ;

    GpiSetColor ( hPS, Alert > 1 ? RGB_RED : ( Alert > 0 ? RGB_YELLOW : RGB_GREEN ) ) ;

    Point.x = 0 ;
    Point.y = 0 ;
    GpiMove ( hPS, &Point ) ;

    Point.x = Rectangle.xRight - 1 ;
    Point.y = Rectangle.yTop - 1 ;
    GpiBox ( hPS, DRO_OUTLINE, &Point, 0L, 0L ) ;

    Point.x = 1 ;
    Point.y = 1 ;
    GpiMove ( hPS, &Point ) ;

    Point.x = Rectangle.xRight - 2 ;
    Point.y = Rectangle.yTop - 2 ;
    GpiBox ( hPS, DRO_OUTLINE, &Point, 0L, 0L ) ;

    Data->Alert = Alert ;
  }
}

/****************************************************************************
 *									    *
 *			    Paint Digital Time				    *
 *									    *
 ****************************************************************************/

static void PaintDigitalTime ( HWND hwnd, HPS hPS, PDATA Data, PDATETIME DateTime )
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  USHORT Hour ;
  RECTL Rectangle ;
  CHAR Text [20] ;

 /***************************************************************************
  * Determine the window rectangle, less the border.			    *
  ***************************************************************************/

  WinQueryWindowRect ( hwnd, &Rectangle ) ;

  Rectangle.xLeft += 2 ;
  Rectangle.xRight -= 2 ;
  Rectangle.yBottom += 2 ;
  Rectangle.yTop -= 2 ;

 /***************************************************************************
  * Draw the new time within the rectangle.				    *
  ***************************************************************************/

  if ( ( DateTime->minutes == 0 ) AND ( DateTime->hours % 12 == 0 ) )
  {
    if ( DateTime->hours == 0 )
    {
      ResourceString Midnight ( Data->Library, IDS_MIDNIGHT ) ;
      strcpy ( Text, PCHAR(Midnight.Ptr()) ) ;
    }
    else if ( DateTime->hours == 12 )
    {
      ResourceString Noon ( Data->Library, IDS_NOON ) ;
      strcpy ( Text, PCHAR(Noon.Ptr()) ) ;
    }
  }
  else
  {
    if ( Data->Profile.Hour24 )
    {
      sprintf ( Text, "%u%s%02u",
	DateTime->hours, Data->CountryInfo.szTimeSeparator, DateTime->minutes ) ;
    }
    else
    {
      ResourceString Am ( Data->Library, IDS_AM ) ;
      ResourceString Pm ( Data->Library, IDS_PM ) ;

      Hour = DateTime->hours % 12 ;

      if ( Hour == 0 )
	Hour = 12 ;

      sprintf ( Text, "%u%s%02u%s",
	Hour, Data->CountryInfo.szTimeSeparator, DateTime->minutes,
	(DateTime->hours>=12) ? Pm.Ptr() : Am.Ptr() ) ;
    }
  }

  WinDrawText ( hPS, strlen(Text), PSZ(Text), &Rectangle,
    Data->Profile.TextColor, Data->Profile.BackColor,
    DT_CENTER | DT_VCENTER | DT_ERASERECT ) ;
}

/****************************************************************************
 *									    *
 *			Hide Window Controls				    *
 *									    *
 ****************************************************************************/

static void HideControls
(
  BOOL fHide,
  HWND FrameWindow,
  HWND hwndSysMenu,
  HWND hwndTitleBar,
  HWND hwndMinMax
)
{
 /***************************************************************************
  * Local Declarations							    *
  ***************************************************************************/

  SWP OldPosition ;
  SWP Position ;
  RECTL Rectangle ;

 /***************************************************************************
  * Get original window position and state.				    *
  ***************************************************************************/

  WinQueryWindowPos ( FrameWindow, &OldPosition ) ;

 /***************************************************************************
  * Restore and hide the window.					    *
  ***************************************************************************/

  WinSetWindowPos ( FrameWindow, NULL, 0, 0, 0, 0, SWP_RESTORE | SWP_HIDE ) ;

 /***************************************************************************
  * Determine client window and location.				    *
  ***************************************************************************/

  WinQueryWindowPos ( FrameWindow, &Position ) ;

  Rectangle.xLeft   = Position.x ;
  Rectangle.xRight  = Position.x + Position.cx ;
  Rectangle.yBottom = Position.y ;
  Rectangle.yTop    = Position.y + Position.cy ;

  WinCalcFrameRect ( FrameWindow, &Rectangle, TRUE ) ;

 /***************************************************************************
  * Hide or reveal the controls windows by changing their parentage.	    *
  ***************************************************************************/

  if ( fHide )
  {
    WinSetParent ( hwndSysMenu,  HWND_OBJECT, FALSE ) ;
    WinSetParent ( hwndTitleBar, HWND_OBJECT, FALSE ) ;
    WinSetParent ( hwndMinMax,	 HWND_OBJECT, FALSE ) ;
  }
  else
  {
    WinSetParent ( hwndSysMenu,  FrameWindow, TRUE ) ;
    WinSetParent ( hwndTitleBar, FrameWindow, TRUE ) ;
    WinSetParent ( hwndMinMax,	 FrameWindow, TRUE ) ;
  }

 /***************************************************************************
  * Tell the frame that things have changed.  Let it update the window.     *
  ***************************************************************************/

  WinSendMsg ( FrameWindow, WM_UPDATEFRAME,
    MPFROMSHORT ( FCF_TITLEBAR | FCF_SYSMENU | FCF_MINBUTTON ), 0L ) ;

 /***************************************************************************
  * Reposition the frame around the client window, which is left be.	    *
  ***************************************************************************/

  WinCalcFrameRect ( FrameWindow, &Rectangle, FALSE ) ;

  WinSetWindowPos ( FrameWindow, NULL,
    (SHORT) Rectangle.xLeft,  (SHORT) Rectangle.yBottom,
    (SHORT) (Rectangle.xRight-Rectangle.xLeft),
    (SHORT) (Rectangle.yTop-Rectangle.yBottom),
    SWP_SIZE | SWP_MOVE ) ;

 /***************************************************************************
  * If window was maximized, put it back that way.			    *
  ***************************************************************************/

  if ( OldPosition.fl & SWP_MAXIMIZE )
  {
    WinSetWindowPos ( FrameWindow, NULL,
      (SHORT) Rectangle.xLeft,	(SHORT) Rectangle.yBottom,
      (SHORT) (Rectangle.xRight-Rectangle.xLeft),
      (SHORT) (Rectangle.yTop-Rectangle.yBottom),
      SWP_SIZE | SWP_MOVE |
      ( OldPosition.fl & SWP_MAXIMIZE ) ) ;
  }

 /***************************************************************************
  * Reveal the window to the curious world.				    *
  ***************************************************************************/

  WinShowWindow ( FrameWindow, TRUE ) ;
}

/****************************************************************************
 *									    *
 *    Monitor Loop Thread						    *
 *									    *
 ****************************************************************************/

static VOID MonitorLoopThread ( PMONITOR_PARMS Parms )
{
 /***************************************************************************
  * Set this thread's priority as high as it can go.                        *
  ***************************************************************************/

  DosSetPrty ( PRTYS_THREAD, PRTYC_TIMECRITICAL, PRTYD_MAXIMUM, 0 ) ;

 /***************************************************************************
  * Start up the high resolution timer, if it is available.		    *
  ***************************************************************************/

  BOOL HiResTimer = OpenTimer ( ) ;

 /***************************************************************************
  * Loop forever . . .							    *
  ***************************************************************************/

  while ( 1 )
  {

   /*************************************************************************
    * Reset the last time and count seen.				    *
    *************************************************************************/

    ULONG LastMilliseconds ;
    TIMESTAMP Time [2] ;

    if ( HiResTimer )
      GetTime ( &Time[0] ) ;
    else
      DosQuerySysInfo ( QSV_MS_COUNT, QSV_MS_COUNT, &LastMilliseconds, sizeof(LastMilliseconds) ) ;

    ULONG LastCounter = *Parms->Counter ;

   /*************************************************************************
    * Sleep for a bit.							    *
    *************************************************************************/

    DosSleep ( 1000 ) ;

   /*************************************************************************
    * Find out how much time and counts went by.			    *
    *************************************************************************/

    ULONG CurrentCounter = *Parms->Counter ;

    ULONG DeltaMilliseconds ;

    if ( HiResTimer )
    {
      GetTime ( &Time[1] ) ;

      ULONG Nanoseconds ;
      DeltaMilliseconds = ElapsedTime ( &Time[0], &Time[1], &Nanoseconds ) ;

      if ( Nanoseconds >= 500000L )
	DeltaMilliseconds ++ ;
    }
    else
    {
      ULONG Milliseconds ;
      DosQuerySysInfo ( QSV_MS_COUNT, QSV_MS_COUNT, &Milliseconds, sizeof(Milliseconds) ) ;
      DeltaMilliseconds = Milliseconds - LastMilliseconds ;
    }

   /*************************************************************************
    * Find out how much idle time was counted.	Adjust it to persecond.     *
    *************************************************************************/

    ULONG Counter = (ULONG) ( ( (double)(CurrentCounter-LastCounter) * 1000L ) / (double)DeltaMilliseconds ) ;

   /*************************************************************************
    * Tell the owner window to refresh its statistics.			    *
    *************************************************************************/

    WinPostMsg ( Parms->Owner, WM_REFRESH, MPFROMLONG(Counter), 0L ) ;
  }
}

/****************************************************************************
 *									    *
 *			 Calibrate the Load Meter			    *
 *									    *
 ****************************************************************************/

static ULONG CalibrateLoadMeter ( void )
{
 /***************************************************************************
  * Set result to zero as a default.					    *
  ***************************************************************************/

  double AdjustedMaxLoad = 0.0 ;

 /***************************************************************************
  * If HRTIMER.SYS has been installed . . .				    *
  ***************************************************************************/

  if ( OpenTimer ( ) )
  {
   /*************************************************************************
    * Increase this thread's priority to the maximum.                       *
    *************************************************************************/

    DosSetPrty ( PRTYS_THREAD, PRTYC_TIMECRITICAL, PRTYD_MAXIMUM, 0 ) ;

   /*************************************************************************
    * Create the calibration thread and set its priority next highest.	    *
    *************************************************************************/

    TID tidCalibrate ;
    ULONG MaxLoad ;
    DosCreateThread ( &tidCalibrate, CounterThread, (ULONG)&MaxLoad, 0, 4096 ) ;
    DosSetPrty ( PRTYS_THREAD, PRTYC_TIMECRITICAL, PRTYD_MAXIMUM-1, tidCalibrate ) ;
    DosSuspendThread ( tidCalibrate ) ;

   /*************************************************************************
    * Reset the calibration count, get the time, and let the counter go.    *
    *************************************************************************/

    MaxLoad = 0 ;
    TIMESTAMP Time[2] ;
    GetTime ( &Time[0] ) ;
    DosResumeThread ( tidCalibrate ) ;

   /*************************************************************************
    * Sleep for one second.						    *
    *************************************************************************/

    DosSleep ( 1000 ) ;

   /*************************************************************************
    * Suspend the calibration counter and get the time. 		    *
    *************************************************************************/

    DosSuspendThread ( tidCalibrate ) ;
    GetTime ( &Time[1] ) ;

   /*************************************************************************
    * Return priorities to normal.					    *
    *************************************************************************/

    DosSetPrty ( PRTYS_THREAD, PRTYC_REGULAR, 0, 0 ) ;

   /*************************************************************************
    * Get the elapsed time and adjust the calibration count.		    *
    *************************************************************************/

    ULONG Milliseconds ;
    ULONG Nanoseconds ;
    Milliseconds = ElapsedTime ( &Time[0], &Time[1], &Nanoseconds ) ;

    AdjustedMaxLoad = (double)MaxLoad * 1.0E9 ;
    AdjustedMaxLoad /= (double)Milliseconds*1.0E6L + (double)Nanoseconds ;

   /*************************************************************************
    * Close down the connection to HRTIMER.SYS. 			    *
    *************************************************************************/

    CloseTimer ( ) ;
  }

 /***************************************************************************
  * Return the adjusted calibration count.  If HRTIMER was not there, it    *
  *   will be zero.							    *
  ***************************************************************************/

  return ( (ULONG)AdjustedMaxLoad ) ;
}

/****************************************************************************
 *									    *
 *		      General Purpose Counter Thread			    *
 *									    *
 ****************************************************************************/

static VOID CounterThread ( PULONG Counter )
{
  while ( 1 )
  {
    (*Counter) ++ ;
  }
}

/****************************************************************************
 *									    *
 *	Open the Profile						    *
 *									    *
 ****************************************************************************/

static HINI OpenProfile ( PSZ Name, HAB Anchor, HMODULE Library, HWND HelpInstance )
{
 /***************************************************************************
  * Query the system INI for the profile file's path.                       *
  ***************************************************************************/

  PSZ ProfilePath = NULL ;
  ULONG Size ;

  if ( PrfQueryProfileSize ( HINI_USERPROFILE, PSZ(PROGRAM_NAME), PSZ("INIPATH"), &Size ) )
  {
    // The info exists.  Fetch it.
    ProfilePath = PSZ ( AllocateMemory ( Size ) ) ;
    PrfQueryProfileData ( HINI_USERPROFILE, PSZ(PROGRAM_NAME), PSZ("INIPATH"),
      ProfilePath, &Size ) ;

    // Build the profile file name.
    BYTE FullPath [_MAX_PATH] ;
    strcpy ( PCHAR(FullPath), PCHAR(ProfilePath) ) ;
    strcat ( PCHAR(FullPath), "\\" ) ;
    strcat ( PCHAR(FullPath), PCHAR(Name) ) ;
    strcat ( PCHAR(FullPath), ".INI" ) ;

    // Clean the name up and expand it to a full path.
    BYTE Path [256] ;
    DosQueryPathInfo ( FullPath, FIL_QUERYFULLNAME, Path, sizeof(Path) ) ;

    // Does the file exist?  If not, discard the name.
    FILESTATUS3 Status ;
    if ( DosQueryPathInfo ( Path, FIL_STANDARD, &Status, sizeof(Status) ) )
    {
      FreeMemory ( ProfilePath ) ;
      ProfilePath = NULL ;
    }
  }

 /***************************************************************************
  * If the profile file couldn't be found, ask the user for a path.         *
  ***************************************************************************/

  if ( ProfilePath == NULL )
  {
    // Set the default path.
    BYTE Path [256] ;
    DosQueryPathInfo ( PSZ("."), FIL_QUERYFULLNAME, Path, sizeof(Path) ) ;

    // Call up the entry dialog.
    PROFILE_PARMS Parms ;
    Parms.id = IDD_PROFILE_PATH ;
    Parms.hwndHelp = HelpInstance ;
    Parms.Path = Path ;
    Parms.PathSize = sizeof(Path) ;
    if ( WinDlgBox ( HWND_DESKTOP, HWND_DESKTOP, ProfileProcessor,
      Library, IDD_PROFILE_PATH, &Parms ) )
    {
      // If OK, save the approved path in the system profile.
      ProfilePath = PSZ ( AllocateMemory ( strlen(PCHAR(Path)) + 1 ) ) ;
      strcpy ( PCHAR(ProfilePath), PCHAR(Path) ) ;

      PrfWriteProfileData ( HINI_USERPROFILE, PSZ(PROGRAM_NAME), PSZ("INIPATH"),
	ProfilePath, strlen(PCHAR(ProfilePath))+1 ) ;
    }
    else
    {
      // If not, return an error.
      return ( NULL ) ;
    }
  }

 /***************************************************************************
  * Build the full profile file name.					    *
  ***************************************************************************/

  BYTE ProfileName [_MAX_PATH] ;
  strcpy ( PCHAR(ProfileName), PCHAR(ProfilePath) ) ;
  strcat ( PCHAR(ProfileName), "\\" PROGRAM_NAME ".INI" ) ;

 /***************************************************************************
  * Release the memory previously allocated to store the path.		    *
  ***************************************************************************/

  if ( ProfilePath )
  {
    FreeMemory ( ProfilePath ) ;
  }

 /***************************************************************************
  * Open/Create the profile file and return the resultant handle.	    *
  ***************************************************************************/

  return ( PrfOpenProfile ( Anchor, ProfileName ) ) ;
}
