/* ------------------------------------------------------------------------ *\
|                                                                            |
|  CORE.H    | <AppName> core definitions.                                   |
|            |                                                               |
|            |                                                               |
|            | Hungarian conventions                                         |
|            | ---------------------                                         |
|            | a  - array                                                    |
|            | b  - boolean char                                             |
|            | c  - single character                                         |
|            | i  - short integer ( 2 bytes )                                |
|            | l  - long integer ( 4 bytes )                                 |
|            | d  - double precision ( 8 bytes )                             |
|            | p  - pointer                                                  |
|            | g  - structure ( [g]roup of elements )                        |
|            | j  - union ( [j]oined elements )                              |
|            | u  - unsigned                                                 |
|            | v  - void                                                     |
|            | sz - string pointer ( zero terminated )                       |
|            | sb - string pointer ( buffered )                              |
|            | ws - WSTR ( hybrid string structure )                         |
|                                                                            |
\* ------------------------------------------------------------------------ */


/* GET/SET definitions ---------------------------------------------------- */

#define GET_ONLY 0
#define GET_SET  1

#define CTYPE_G     0      // struct
#define CTYPE_C   256      // char
#define CTYPE_I   512      // short
#define CTYPE_B  1024      // bool
#define CTYPE_L  2048      // long
#define CTYPE_D  4096      // double
#define CTYPE_S  8192      // *char

#define GET_CTYPE 16128    // logical AND with handle to get C data type

#define GETSET_C(iHnd,cVal)  ( * ( (char*)   _vpXXGetSet ( iHnd , &cVal ) ) )
#define GETSET_B(iHnd,bVal)  ( * ( (bool*)   _vpXXGetSet ( iHnd , &bVal ) ) )
#define GETSET_I(iHnd,iVal)  ( * ( (short*)  _vpXXGetSet ( iHnd , &iVal ) ) )
#define GETSET_L(iHnd,lVal)  ( * ( (long*)   _vpXXGetSet ( iHnd , &lVal ) ) )
#define GETSET_D(iHnd,dVal)  ( * ( (double*) _vpXXGetSet ( iHnd , &dVal ) ) )
#define GETSET_S(iHnd,sVal)  (     (WSTR*)   _vpXXGetSet ( iHnd ,  sVal )   )

// NOTE: the following constants are "handles" to various GET/SET items.
//       The high order values in CTYPE_? are added to the low order values 
//       of each handle to codify the C data type.  The function 
//       __vpXXGetSet() decodes these handles in order to GET/SET within the
//       proper C data type.

#define HND_ERR_FPTR         0 + CTYPE_L
#define HND_ERR_FNAME        1 + CTYPE_S



/* Error definitions ------------------------------------------------------ */

#define EC_MAX           40  // up to 40 error codes
#define EC_LEN           41  // of 40 char each

#define EC_NONE           0  // successful
#define EC_DEMO           1  // demo limit has been exceeded
#define EC_WR_PRN         2  // unable to write to print device
#define EC_HANDLER        3  // unable to call error handler
#define EC_GSVALUE        4  // internal error (GET/SET handle)
#define EC_GSCTYPE        5  // internal error (C data type in GET/SET)
#define EC_RANGE          6  // parameter is out of range 
#define EC_TYPE           7  // parameter is invalid data type
#define EC_COUNT          8  // invalid parameter count
#define EC_EC             9  // error code is invalid
#define EC_ALLOC         10  // unable to allocate memory from API
#define EC_GDI_DC        11  // unable to get HDC for windows printing
#define EC_GDI_CMD       12  // unable to execute GDI command
#define EC_FNAME         13  // unable to open named file



/* Interface core error handling to WRAPI --------------------------------- */

#define SYSTEM_NAME()     "<AppName>"

#define IS_ERR_NONE()     ( _iXXErrCode ( NIL_I ) == EC_NONE )

#define IS_ERR_ERR()      ( _iXXErrCode ( NIL_I ) == EC_EC )

#define GET_ERR_CODE()    ( _iXXErrCode ( NIL_I ) )

#define GET_ERR_FPTR()    ( * ( (long*) _vpXXGetSet ( HND_ERR_FPTR ,  NULL ) ) )

#define GET_ERR_FNAME()   ( ( * ( (WSTR*) _vpXXGetSet ( HND_ERR_FNAME , NULL ) ) ) .cp )

#define GET_ERR_HOST()    ( _iXXErrHost ( NIL_I ) )

#define GET_ERR_LOC()     ( ( * _wspXXErrLoc ( NIL_G ) ) .cp )

#define GET_ERR_TEXT(i)   ( ( * _wspXXErrText ( i , NIL_G ) ) .cp )

#define SET_ERR_NONE()    ( _iXXErrCode ( EC_NONE ) )

#define SET_ERR_TYPE()    ( _iXXErrCode ( EC_TYPE ) )

#define SET_ERR_COUNT()   ( _iXXErrCode ( EC_COUNT ) )

#define SET_ERR_ALLOC()   ( _iXXErrCode ( EC_ALLOC ) )

#define SET_ERR_OUT()     ( _iXXErrCode ( EC_WR_PRN ) )

#define SET_ERR_GDI_CMD() ( _iXXErrCode ( EC_GDI_CMD ) )

#define SET_ERR_LOC(sz)   ( _wspXXErrLoc ( ztow ( sz ) ) )

#define CALL_ERR_VBD()    XXErrEvent ( )

#define CALL_ERR_CLIP4()  XXErrEvent ( )

#define CALL_ERR_MSG()    _szXXErrMsg ( )



/* Macros used by the GET/SET engine -------------------------------------- */

#define IS_RANGE(n,l,u)  ( (bool) ( n>=l && n<=u ) )

#define IS_GTE_ZERO(d)   ( (bool) ( d>=0.0 ) )

#define CHECK_ARG_RANGE(b) if ( bIsNew && ! ( b ) ) \
                             gXXErr.iCode = EC_RANGE ;

#define GETSET(pOld,New) if ( gXXErr.iCode == EC_GSVALUE )                \
                           MEMSET ( vpGet , '0' , sizeof ( New ) ) ;      \
                         else {                                           \
                           MEMCPY ( vpGet , pOld , sizeof ( New ) )  ;    \
                           if ( bIsNew && !gXXErr.iCode ) *pOld = New ; }



/* Typedefs --------------------------------------------------------------- */

typedef struct {                   // ERROR
  short     iCode                ; // error value
  char      szaLoc          [16] ; // error location
  long      lFPtr                ; // pointer to 'C' error handler
  char      szaFName        [11] ; // name of FoxPro error handler
  short     iHost                ; // host language error number
  short     iOS                  ; // OS error code
} ERRDEF ;



/* Function prototypes ---------------------------------------------------- */


#ifdef _FAMILY_C

  // Convert API PARMLIST to formal 'C' parameters

  #define XXErrCode(PARMLIST)  XXErrCode  ( short iErrCode )
  #define XXErrLoc(PARMLIST)   XXErrLoc   ( WSP wspErrLoc )
  #define XXErrText(PARMLIST)  XXErrText  ( short iErrCode , WSP wspErrText )
  #define XXErrHost(PARMLIST)  XXErrHost  ( short iErrHost )
  #define XXErrFPtr(PARMLIST)  XXErrFPtr  ( long lFuncPtr )
  #define XXErrFName(PARMLIST) XXErrFName ( WSP wspFuncName )
  #define XXErrQOS(PARMLIST)   XXErrQOS   ( void )

#endif

// Wrapper functions

TYPE_I XXErrCode  ( PARMLIST ) ;
TYPE_S XXErrLoc   ( PARMLIST ) ;
TYPE_S XXErrText  ( PARMLIST ) ;
TYPE_I XXErrHost  ( PARMLIST ) ;
TYPE_L XXErrFPtr  ( PARMLIST ) ;
TYPE_S XXErrFName ( PARMLIST ) ;
TYPE_I XXErrQOS   ( PARMLIST ) ;


// Core functions

short    _iXXErrCode         ( short )                                       ;
short    _iXXErrHost         ( short )                                       ;
short    _iXXErrQOS          ( void )                                        ;
WSTR   * _wspXXErrLoc        ( WSTR* )                                       ;
WSTR   * _wspXXErrText       ( short, WSTR* )                                ;
char   * _szXXErrMsg         ( void )                                        ;

void   * _vpXXGetSet         ( short , void* )                               ;



/* End-of-file ------------------------------------------------------------ */

