/*
**      DDEIO.PRG
**
**      This module contains Dynamic Data Exchage support for clipper
**      programs using the DDEDRV device driver.
**
**      (c) 1993 Copyright Portable Software, Inc.
**      All Rights Reserved.
**
**      This source code is provided "as is", with no warranties expressed
**      or implied.
*/

#include "fileio.ch"

/* error codes */

#define         ERR_AGENT_ACK           0
#define         ERR_ACK_ID              1
#define         FIRST_ERROR             10
#define         ERR_AGENT_NACK          10
#define         ERR_IN_USE              11
#define         ERR_TIMEOUT             12
#define         ERR_INVALID_COMMAND     13
#define         ERR_PARM_TYPE           14
#define         ERR_PARM_ERROR          15
#define         ERR_DRIVER              16
#define         ERR_COMMAND_LINE        17
#define         ERR_UNKNOWN_PARM        18        
#define         ERR_INTERNAL            23

/* misc defines */

#define         DDE_DEFAULT_WAIT        20


/****************************DDEInitiate**********************************
**
** Initiate a conversation with a Windows program.
**
** Input :
** Application name and topic name to communicate with.
** The waittime value represents seconds and is used to timeout 
** the request.
**
** Output :
** The value returned is a number between 1 or 8 if the call 
** is successful.  A number between 1 and 8 indicates an open channel
** through which all DDE verbs can be used.  Before any other verbs
** can be used, i.e. DDEPoke, DDERequest, DDEExecute or DDETerminate
** a successful DDEInitiate call must be performed.
** 
** Example:    channel = DDEInitiate("PROGMAN", "PROGMAN", 30)
**
** This example opens up a channel with the Windows program 
** manager.
************************************************************************/


FUNCTION DDEInitiate(app, topic, waittime)
LOCAL nHandle, timeout, rc

IF app = NIL .OR. topic = NIL 
   RETURN ERR_PARM_ERROR 
ENDIF

IF VALTYPE(app) != "C" .OR. VALTYPE(topic) != "C"
   RETURN ERR_PARM_TYPE
ENDIF

IF waittime = NIL
   waittime = DDE_DEFAULT_WAIT
ENDIF

/* open the dde device */

nHandle = FOPEN("ddedrv", FO_READWRITE)
IF FERROR() != 0 
        RETURN ERR_DRIVER
ENDIF


/* write out the control registers */

IOCTL_OUT(nHandle, "v=INITIATE ")
IOCTL_OUT(nHandle, "a=" + app + " ")
IOCTL_OUT(nHandle, "t=" + topic + " ")


/* wait for completion */


rc = IOCTL_IN(nHandle) 
timeout = SECONDS()

DO WHILE ABS(SECONDS() - timeout) <= waittime  .AND. rc == "12" 
         rc = IOCTL_IN(nHandle) 
ENDDO


/* close device and return result. */

FCLOSE(nHandle)
RETURN VAL(rc)



/****************************DDETerminate**********************************
**
** Terminate a conversation with a Windows program.
**
** Input :
** An open DDE channel number
** A timeout value.
**
** Output :
** error_code Success or Failure.
** 
** Example:    error_code = DDETerminate(channel, 30)
**
** This example closes a channel with a Windows program.
************************************************************************/


FUNCTION DDETerminate(channel, waittime)
LOCAL nHandle, timeout, rc

IF channel = NIL  
   channel = 1 
ENDIF


IF waittime = NIL
   waittime = DDE_DEFAULT_WAIT
ENDIF


IF VALTYPE(channel) != "N" .OR. VALTYPE(waittime) != "N"
   RETURN ERR_PARM_TYPE
ENDIF


/* open the dde device */

nHandle = FOPEN("ddedrv", FO_READWRITE)
IF FERROR() != 0 
        RETURN ERR_DRIVER
ENDIF


/* write out the control registers */

IOCTL_OUT(nHandle, "v=TERMINATE ")
IOCTL_OUT(nHandle, "c=" + STR(channel) + " ")


/* wait for completion */

timeout = SECONDS()
rc = IOCTL_IN(nHandle)
DO WHILE ABS(SECONDS() - timeout) <= waittime .AND. rc == "12"
        rc = IOCTL_IN(nHandle)   
ENDDO

/* close device and return result. */

FCLOSE(nHandle)
RETURN VAL(rc)


/****************************DDEExecute**********************************
**
** Perform an Execute through an open channel.
**
** Input :
** An open DDE channel number
** A command string to execute
** A timeout value.
**
** Output :
** error_code Success or Failure.
** 
** Example:  error_code = DDEExecute(channel, "[CreateGroup(Clipper)]", 30)
**
** This example sends the command "[CreateGroup(Clipper)]" to a Windows
** program.  If the channel was opened with the Windows program manager
** this would result in a new program group being added.
************************************************************************/


FUNCTION DDEExecute(channel, command_str, waittime)
LOCAL nHandle, timeout, rc

IF command_str = NIL
    RETURN DDE_PARM_ERROR
ENDIF

IF VALTYPE(command_str) != "C"
    RETURN DDE_PARM_TYPE
ENDIF

IF channel = NIL  
   channel = 1 
ENDIF


IF waittime = NIL
   waittime = DDE_DEFAULT_WAIT
ENDIF


IF VALTYPE(channel) != "N" .OR. VALTYPE(waittime) != "N"
   RETURN ERR_PARM_TYPE
ENDIF


/* open the dde device */

nHandle = FOPEN("ddedrv", FO_READWRITE)
IF FERROR() != 0 
        RETURN ERR_DRIVER
ENDIF


/* write out the control registers */

IOCTL_OUT(nHandle, "v=EXECUTE ")
IOCTL_OUT(nHandle, "c=" + STR(channel) + " ")
IOCTL_OUT(nHandle, "i=" + command_str + " ")


/* wait for completion */

timeout = SECONDS()
rc = IOCTL_IN(nHandle)
DO WHILE ABS(SECONDS() - timeout) <= waittime .AND. rc == "12"
        rc = IOCTL_IN(nHandle)   
ENDDO

/* close device and return result. */

FCLOSE(nHandle)
RETURN VAL(rc)



/****************************DDERequest**********************************
**
** Request an item from an open channel.
**
** Input :
** An open DDE channel number
** An item name
** A buffer to receive the item's value 
** A timeout value.
**
** Output :
** error_code Success or Failure.
** buffer is filled with item's value
** 
** Example:  error_code = DDERequest(channel, "Groups", groups, 30)
**
** This example reqeusts the value of the item named "Groups" from a 
** Windows program using the open channel.
** If the channel was opened with the Windows program manager
** this request would result with the buffer being filled with
** a list of all the program group names.
************************************************************************/


FUNCTION DDERequest(channel, item, buffer, waittime)
LOCAL nHandle, timeout, rc, byte

IF item = NIL  .OR. buffer = NIL
    RETURN DDE_PARM_ERROR
ENDIF

IF VALTYPE(item) != "C"
    RETURN DDE_PARM_TYPE
ENDIF

IF channel = NIL  
   channel = 1 
ENDIF


IF waittime = NIL
   waittime = DDE_DEFAULT_WAIT
ENDIF


IF VALTYPE(channel) != "N" .OR. VALTYPE(waittime) != "N"
   RETURN ERR_PARM_TYPE
ENDIF


/* open the dde device */

nHandle = FOPEN("ddedrv", FO_READWRITE)
IF FERROR() != 0 
        RETURN ERR_DRIVER
ENDIF


/* write out the control registers */

IOCTL_OUT(nHandle, "v=REQUEST ")
IOCTL_OUT(nHandle, "c=" + STR(channel) + " ")
IOCTL_OUT(nHandle, "i=" + item + " ")


/* wait for completion */

timeout = SECONDS()
rc = IOCTL_IN(nHandle)
DO WHILE ABS(SECONDS() - timeout) <= waittime .AND. rc == "12"
        rc = IOCTL_IN(nHandle)   
ENDDO

IF rc == "0"
   byte = FREADSTR(nHandle, 1)
   DO WHILE ASC(byte) != 0
       buffer = buffer + byte
       byte = FREADSTR(nHandle, 1)
   ENDDO
ENDIF

/* close device and return result. */

FCLOSE(nHandle)
RETURN VAL(rc)


/****************************DDEPoke**********************************
**
** Poke a value into an item through an open channel.
**
** Input :
** An open DDE channel number
** An item name
** The item's new value 
** A timeout value.
**
** Output :
** error_code Success or Failure.
** 
** Example:  error_code = DDEPoke(channel, "R1C1", cell, 30)
**
** This example pokes the item "R1C1" with the value cell.  If the channel
** opened with an Excel spreadsheet, this DDEPoke would update the value
** of a spreadsheet cell.
************************************************************************/


FUNCTION DDEPoke(channel, item, value, waittime)
LOCAL nHandle, timeout, rc, i

IF item = NIL  .OR. value = NIL
    RETURN DDE_PARM_ERROR
ENDIF

IF VALTYPE(item) != "C"  .AND. VALTYPE(value) != "C"
    RETURN DDE_PARM_TYPE
ENDIF

IF channel = NIL  
   channel = 1 
ENDIF


IF waittime = NIL
   waittime = DDE_DEFAULT_WAIT
ENDIF


IF VALTYPE(channel) != "N" .OR. VALTYPE(waittime) != "N"
   RETURN ERR_PARM_TYPE
ENDIF


/* open the dde device */

nHandle = FOPEN("ddedrv", FO_READWRITE)
IF FERROR() != 0 
        RETURN ERR_DRIVER
ENDIF


/* write out the control registers */

IOCTL_OUT(nHandle, "v=POKE ")
IOCTL_OUT(nHandle, "c=" + STR(channel) + " ")
IOCTL_OUT(nHandle, "i=" + item + " ")


/* write out the value as a sequence of bytes */

FOR i:= 1 to LEN(value)
   FWRITE(nHandle, SUBSTR(value, i, 1), 1)
NEXT

/* wait for completion */

timeout = SECONDS()
rc = IOCTL_IN(nHandle)
DO WHILE ABS(SECONDS() - timeout) <= waittime .AND. rc == "12"
        rc = IOCTL_IN(nHandle)   
ENDDO

/* close device and return result. */

FCLOSE(nHandle)
RETURN VAL(rc)

