//***************************************************************************
//*                                                                         *
//* Interactive 3D Kit                                                      *
//*                                                                         *
//* I3D API                                                                 *
//*                                                                         *
//* (c) 1993 Jim O'Keane                                                    *
//* All Rights Reserved Worldwide                                           *
//*                                                                         *
//***************************************************************************

#ifndef _I3DKIT_H

#define _I3DKIT_H 1

// Simple macros to facilitate global vars in .h files
// Define MAIN in only one file before #includes.
#ifndef MAIN
#define GLOBAL extern
#else
#define GLOBAL
#endif

// MS Windows-like types
#ifndef TRUE
typedef unsigned char     BYTE;
typedef unsigned short    WORD;
typedef unsigned long     DWORD;
typedef short		    BOOL;
#define FALSE		    0
#define TRUE		    1
#endif

#ifndef FAR
#define FAR far
#endif

#ifndef PASCAL
#define PASCAL pascal
#endif

#ifdef __FLAT__
// If 32 bit compiler, dump the "far" and the "pascal"

#undef  FAR
#define FAR  

#undef  PASCAL
#define PASCAL

#endif

// pointer to the 3D Window buffer, alloc'd by your code
GLOBAL BYTE FAR *winBuf;

// Maps are always 128 by 128 blocks. (128 x 128 x 2 = 32K Bytes)
// If you need a smaller map, just don't use whole array.
// The map contains indexes into the block definition array.
#define MAP_WIDTH  128
#define MAP_HEIGHT 128

// Map array type
typedef short MAPTYPE[MAP_WIDTH][MAP_HEIGHT];

// size of map blocks (high byte of coords is map coord)
#define BLOCK_SIZE     256
// mask is used to find position inside a block
#define BLOCK_MASK     255
// shift is used to find the map coords
#define BLOCK_SHIFT    8


// Block shape types 
#define BLOCK_EMPTY    0  // block is empty (just floor & ceiling)
#define BLOCK_CUBE     1  // block is a cube
#define BLOCK_HORZ     2  // block is a horizontal (EW) divider
#define BLOCK_VERT     3  // block is a vertical (NS) divider
#define BLOCK_ACTOR    4  // block is an actor or prop (a thing)

// Bit flags
#define BLOCK_TRANS    1   // block is transparent
#define BLOCK_WALL     2   // block is an impassable

// definition of a block
typedef struct tag_block
{
  short ns_wall;   // wall panel to show (-1 = none)
  short ew_wall;   // wall panel to show (-1 = none)
  short ceil;      // ceiling panel to show
  short floor;     // floor panel to show
  short shape;     // type of shape
  short flags;     // bit flags
  short x_offset;  // positional offset (+/- 1/2 BLOCK_SIZE)
  short y_offset;  // used to animate blocks and things
  short user1;
  short user2;     // free for user defined uses
  short user3;
  short user4;
} BLOCK;

// if a block is an actor or prop, cast the block struct to this:
typedef struct tag_thing 
{
  short panel;     // base panel to show (-1 = none)
  short block;     // what background block is at this location
  short views;     // how many directions give different views?
  short heading;   // direction facing
  short shape;     // type of shape
  short flags;     // bit flags
  short x_offset;  // positional offset +/- 1/2 BLOCK_SIZE
  short y_offset;  // used to animate blocks and things
  short user1;
  short user2;     // free for user defined uses
  short user3;
  short user4;
} THING;

// standard image operator macros
#define IMAGE_SIZE256(w,h) ((unsigned short)((long)(w)*(h)))
#define PIXPOS(x,y,w) (((y)*(w))+(x))

// handy macros
#define MIN(a,b) ((a) < (b) ? (a) : (b))
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#define SIGN(x)  (((x) < 0) ? -1 : (((x) > 0) ? 1 : 0))
#define ABS(x) ((x)<0 ? -(x):(x))

// Prototypes ////////////////////////////////////////////////////

// Quicky integer math functions. High speed, low accuracy.
short FAR q_sin( short angle );
short FAR q_cos( short angle );
void  FAR rotate(short *x, short *y, short cx, short cy, short angle);

// I3D functions ////////////////////////////

// Tell the I3D engine what map to use. I3D keeps a copy of
// this pointer, so you can change the map on the fly.

BOOL  FAR i3d_set_map(MAPTYPE FAR *map_ptr);

// Tell the I3D engine about the texture map panels. I3D keeps a copy of
// the pointer to the panel list, so you can change the texture maps on
// the fly.

BOOL  FAR i3d_set_panels(short num_panels, BYTE FAR **panel_list,
                         short size,short shift);

// Tell the I3D engine about you block definition list. I3D keeps a copy
// of the pointer to the block list, so you can change the block definitions
// on the fly.
                      
BOOL  FAR i3d_set_blocks(short num_blocks, BLOCK FAR *block_list);

// Tell the I3D engine to use a solid color for floors and ceilings. If set
// to zero, uses texture mapped floors and ceilings. 

BOOL  FAR i3d_set_floor_ceil(short floor_col, short ceil_col);

// Tell the I3D engine the dimensions of the window buffer it uses, and
// the width of a line in bytes.

BOOL  FAR i3d_set_window_buffer(short width, short height,
                                short line_width, BOOL invert,
                                short scale, BYTE FAR *buf);

// Ask I3D to create a frame in the window buffer.

void FAR i3d_view_scan256(short viewer_x, short viewer_y,
                          short eye_angle,
                          short viewer_heading);

// Ask I3D to tell you what is hit at a particular point in the window buffer.

void FAR i3d_hit_scan256(short viewer_x, short viewer_y,
                         short eye_angle,
                         short viewer_heading,
                         short click_x, short click_y,
                         short FAR *h_x, short FAR *h_y, short FAR *h_z,
                         short FAR *block_id, short FAR *panel_u, short FAR *panel_v);
#endif
